﻿using Curse.Friends.NotificationContracts;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using System.Security;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Events;
using Curse.FriendsService.Tester.Utilities;

namespace Curse.FriendsService.Tester.ConsoleApp
{
    internal class Program
    {
        private static Guid _currentGroupID = Guid.Empty;

        // This is the current call, regardless of whether any invites have been received before or after joining
        private static string _currentCallInviteCode = null;

        // These are all used to determine which call to join
        private static int? _currentVoiceInviteUserID = 0;
        private static Guid? _currentVoiceInviteGroup = Guid.Empty;
        private static string _currentVoiceInviteUrl;
        private static long? _currentVoiceAccessToken;


        private static readonly Dictionary<int, ServiceEnvironment> ServiceUrls = Enum.GetValues(typeof (ServiceEnvironment)).Cast<ServiceEnvironment>().ToDictionary(e => (int) e);

        private static SecureString GetPassword()
        {
            SecureString pwd = new SecureString();
            while (true)
            {
                ConsoleKeyInfo i = Console.ReadKey(true);
                if (i.Key == ConsoleKey.Enter)
                {
                    break;
                }
                else if (i.Key == ConsoleKey.Backspace)
                {
                    if (pwd.Length > 0)
                    {
                        pwd.RemoveAt(pwd.Length - 1);
                        Console.Write("\b \b");
                    }
                }
                else
                {
                    pwd.AppendChar(i.KeyChar);
                    Console.Write("*");
                }
            }
            return pwd;
        }

        private static HashSet<int> PromptUserList(int limit = 0)
        {
            var recipientIDs = new HashSet<int>();

            while (true)
            {
                Console.WriteLine("Enter a friend ID to invite (or blank to continue): ");
                var input = Console.ReadLine();
                if (string.IsNullOrEmpty(input))
                {
                    break;
                }
                recipientIDs.Add(int.Parse(input));

                if (recipientIDs.Count == limit)
                {
                    break;
                }
            }

            return recipientIDs;
        }

        private static void Divider()
        {
            Console.WriteLine();
            Console.WriteLine("------------------------------------------");
            Console.WriteLine();
        }

        private static TestClient _testClient;

        private static void Main()
        {
            ServicePointManager.ServerCertificateValidationCallback += (a, b, c, d) => true;
            Console.Title = "Friends Tester";

            var environment = SelectServer();

            _testClient = new TestClient(new ServiceConnectionInfo(environment));

            _testClient.Events.Register<MessageSent>(_notificationClient_InstantMessageResponse);
            _testClient.Events.Register<FriendMessageReceived>(_notificationClient_InstantMessageNotification);
            _testClient.Events.Register<GroupMessageReceived>(FriendsNotificationClient_GroupMessageNotification);

            _testClient.Events.Register<FriendshipChanged>(_notificationClient_FriendshipChangeNotification);
            _testClient.Events.Register<FriendshipRemoved>(_notificationClient_FriendshipRemovedNotification);
            _testClient.Events.Register<UserChanged>(Instance_UserChangeNotification);
            _testClient.Events.Register<GroupChanged>(FriendsNotificationClientOnGroupChangeNotification);

            _testClient.Events.Register<OfflineMessagesSynced>(FriendsNotificationClientOnOfflineMessageNotification);

            _testClient.Events.Register<FriendCallInvitationReceived>(FriendsNotificationClient_VoiceInvitationNotification);
            _testClient.Events.Register<GroupCallInvitationReceived>(FriendsNotificationClient_GroupVoiceInvitationNotification);
            _testClient.Events.Register<CallAccepted>(FriendsNotificationClient_VoiceAcceptNotification);
            _testClient.Events.Register<FriendCallDeclined>(FriendsNotificationClient_VoiceDeclineNotification);
            _testClient.Events.Register<GroupCallDeclined>(FriendsNotificationClient_VoiceDeclineNotification);

            _testClient.Events.Register<JoinedCall>(VoiceRuntimeClient_Connected);
            _testClient.Events.Register<LeftCall>(VoiceRuntimeClient_Disconnected);


            ExecutionLoop(GetOptions());
        }

        private static ServiceEnvironment SelectServer()
        {
            ServiceEnvironment url;
            while (true)
            {
                Console.WriteLine("Choose an endpoint:");
                foreach (var kvp in ServiceUrls)
                {
                    Console.WriteLine(kvp.Key + ". " + kvp.Value.GetDescription());
                }

                var resp = Console.ReadKey(true);

                var key = int.Parse(resp.KeyChar.ToString());


                if (ServiceUrls.TryGetValue(key, out url))
                {
                    Console.Clear();
                    break;
                }
            }

            return url;
        }

        private static Dictionary<string, Action> GetOptions()
        {
            var options = new Dictionary<string, Action>();
            options.Add("login", Login);

            // Group commands
            options.Add("new-group", CreateGroup);
            options.Add("leave-group", LeaveGroup);
            options.Add("delete-group", DeleteGroup);
            options.Add("remove-group-member", RemoveGroupMember);
            options.Add("add-group-member", AddGroupMembers);
            options.Add("change-member-role", ChangeGroupMemberRole);
            options.Add("new-group-invite", CreateGroupInvite);
            options.Add("claim-group-invite", ProcessGroupInvite);
            options.Add("delete-group-invite", DeleteGroupInvite);

            options.Add("message-group", GroupMessage);
            options.Add("call-group", CallGroup);
            options.Add("rename-group", RenameGroup);
            options.Add("mute-group", MuteGroup);
            options.Add("unmute-group", UnmuteGroup);
            options.Add("favorite-group", ToggleGroupFavorite);
            options.Add("enter-group", EnterGroup);
            options.Add("offline-messages", SyncOfflineMessages);

            // Self commands
            options.Add("change-status", ChangeStatus);
            options.Add("change-game-status", ChangeGameStatus);
            options.Add("instant-message", InstantMessage);
            options.Add("send-friend-request", SendFriendRequest);
            options.Add("rename-friend", RenameFriend);
            options.Add("confirm-friend-request", ConfirmFriendRequest);
            options.Add("decline-friend-request", DeclineFriendRequest);
            options.Add("favorite-friend", FavoriteFriend);
            options.Add("remove-friend", RemoveFriend);
            options.Add("unblock-friend", UnblockFriend);
            options.Add("get-profile", GetProfile);
            options.Add("search", Search);

            options.Add("call-friend", CallFriend);
            options.Add("decline-call", DeclineVoiceInvite);
            options.Add("accept-call", AcceptVoiceInvite);
            options.Add("hang-up", HangUp);

            return options;
        }

        private static void PrintFriendsAndGroups()
        {
            if (_testClient.IsLoggedIn)
            {
                Console.WriteLine("You are logged in as " + _testClient.Username + "#" + _testClient.UserID);

                GroupCache currentGroup = _testClient.GroupCache.Groups.FirstOrDefault(g => g.GetGroup(_currentGroupID) != null);

                if (currentGroup != null)
                {
                    Console.WriteLine("You are currently in " + currentGroup.RootGroup.GroupTitle);

                    string error;
                    if (!currentGroup.HasLoadedDetails && !currentGroup.LoadDetails(out error))
                    {
                        Console.WriteLine("Failed to load Group Details: " + error);
                    }
                }

                Console.WriteLine();
                Console.WriteLine("GROUPS");


                foreach (var groupCache in _testClient.GroupCache.Groups)
                {
                    Console.WriteLine();
                    Console.WriteLine(groupCache.RootGroup.GroupTitle);// + " (" + groupCache.RootGroup.GroupAccessLevel + "+)");

                    if (!groupCache.HasLoadedDetails)
                    {
                        Console.WriteLine("Details not loaded. Enter this group to load details.");
                    }
                    else
                    {
                        Console.WriteLine("Members");
                        foreach (var member in groupCache.Members.Values)
                        {
                            Console.WriteLine(" -" + member.Username + " (" + member.BestRole + ")");
                        }

                        var children = groupCache.GetChildren();
                        if (children.Any())
                        {
                            Console.WriteLine("Sub Groups");

                            foreach (var child in children.Where(p => p.ParentGroupID == groupCache.RootGroup.GroupID).OrderBy(p => p.DisplayOrder))
                            {
                                Console.WriteLine(" -" + child.GroupTitle);// + " (" + child.GroupAccessLevel + "+)");
                                foreach (var subChild in children.Where(p => p.ParentGroupID == child.GroupID).OrderBy(p => p.DisplayOrder))
                                {
                                    Console.WriteLine("     -" + subChild.GroupTitle);// + " (" + subChild.GroupAccessLevel + "+)");
                                }
                            }
                        }
                    }

                    Console.WriteLine("-------------------------------------------");
                }

                Console.WriteLine();
                Console.WriteLine(_testClient.FriendCache.Friends.Length == 0 ? "You have no friends :(" : "FRIENDS");

                foreach (var group in _testClient.FriendCache.Friends.GroupBy(p => p.FriendshipStatus))
                {
                    Console.WriteLine();
                    Console.WriteLine(group.Key.ToString());

                    var ordered = group.OrderByDescending(p => p.Favorite);

                    foreach (var friend in ordered.Take(10))
                    {
                        Console.Write(" ");
                        if (friend.Favorite)
                        {
                            Console.Write("* ");
                        }
                        Console.WriteLine((friend.Nickname ?? friend.Username) + "#"
                                          + friend.UserID
                                          + " (" + friend.ConnectionStatus
                                          + (!string.IsNullOrEmpty(friend.StatusMessage) ? " - " + friend.StatusMessage : "") + ")");
                    }
                    if (group.Count() > 10)
                    {
                        Console.WriteLine("Showing first 10 friends...");
                    }
                }

            }

        }

        private static void ExecutionLoop(Dictionary<string, Action> options)
        {
            while (true)
            {
                PrintFriendsAndGroups();
                Divider();
                Console.WriteLine("Current Call: " + _currentCallInviteCode ?? "None");
                Divider();

                foreach (var option in options.Keys)
                {
                    Console.WriteLine(" " + option);
                }
                Divider();
                Console.WriteLine("Enter an option from the list above...");

                var entered = Console.ReadLine().ToLower().Trim();
                Console.Clear();

                try
                {
                    if (entered == string.Empty)
                    {
                        Console.Clear();
                        continue;
                    }
                    Action action;
                    if (options.TryGetValue(entered, out action))
                    {
                        action();
                        Console.WriteLine("Press any key to continue...");
                    }
                    else
                    {
                        Console.WriteLine("Unknown option: " + entered);
                    }
                    Console.ReadKey(true);
                }
                catch (Exception)
                {
                    Console.WriteLine("Invalid entry!");
                }

                Console.Clear();

            }
        }

        static void VoiceRuntimeClient_Disconnected(LeftCall value)
        {
            Console.WriteLine("Call disconnected. Initiator: {0}. Reason: {1}", value.InitiatingUserID, value.Reason);
        }

        static void VoiceRuntimeClient_Connected(JoinedCall value)
        {
            _currentCallInviteCode = value.InviteCode;
            Console.WriteLine("Call connected.");
        }


        private static void FriendsNotificationClient_VoiceAcceptNotification(CallAccepted value)
        {
            Console.WriteLine("Voice Invite {0} accepted by {1} at {2}", value.InviteUrl, value.SenderID, value.Timestamp);
        }

        private static void CreateGroup()
        {
            Console.WriteLine("Not Implemented as of refactor!");
            Console.ReadLine();
            //Console.Write("Enter a title: ");
            //var title = Console.ReadLine();


            //Console.Write("Choose a type (Large, normal, temp): ");
            //var typeString = Console.ReadLine();
            //GroupType type;
            //Enum.TryParse(typeString, true, out type);


            //Guid parentGroupID = Guid.Empty;
            //var accessLevel = GroupRole.Member;

            //if (type == GroupType.Large || type == GroupType.Temporary)
            //{
            //    Console.Write("Enter an access level (Member, Officer, Owner leave blank for default): ");
            //    var accessLevelString = Console.ReadLine();
            //    if (!string.IsNullOrEmpty(accessLevelString))
            //    {
            //        Enum.TryParse(accessLevelString, true, out accessLevel);
            //    }

            //    Console.WriteLine("Is this a sub group? (Y|N)");
            //    var isSubGroup = Console.ReadKey(true).Key == ConsoleKey.Y;

            //    if (isSubGroup)
            //    {
            //        parentGroupID = PromptGroup(false);
            //    }
            //}

            //var recipientIDs = PromptUserList();

            //string error;
            //Guid id;
            //if (_testClient.Execute(new CreateGroup(title, type) {ParentGroupID = parentGroupID, AccessLevel = accessLevel, NewMembers = recipientIDs.ToArray()}, out id, out error))
            //{
            //    Console.WriteLine("Group ID: " + id);
            //}
            //else
            //{
            //    Console.WriteLine(error);
            //}
        }

        private static void DeleteGroup()
        {

            var groupID = PromptGroup(false);

            string error;
            if (_testClient.Execute(new DeleteGroup(groupID), out error))
            {
                Console.WriteLine("Deleted group {0}", groupID);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static Guid PromptGroup(bool rootOnly = true)
        {
            var counter = 0;
            var choices = new Dictionary<int, Guid>();
            foreach (var group in _testClient.GroupCache.Groups)
            {
                choices[++counter] = group.RootGroup.GroupID;
                Console.WriteLine(counter + ". " + group.RootGroup.GroupTitle);
                if (!rootOnly)
                {
                    foreach (var child in group.GetChildren())
                    {
                        choices[++counter] = child.GroupID;
                        Console.WriteLine(counter + ". " + child.GroupTitle);
                    }
                }
            }

            Console.Write("Enter the number of the group: ");

            try
            {
                var choice = int.Parse(Console.ReadLine());
                return choices[choice];
            }
            catch (Exception)
            {

                Console.WriteLine("Invalid selection! Press any key to continue...");
                Console.ReadKey(true);
                return Guid.Empty;
            }

        }

        private static void LeaveGroup()
        {
            var group = _currentGroupID;
            string error;
            if (_testClient.Execute(new LeaveGroup(group), out error))
            {
                Console.WriteLine("Left group {0}", group);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void RemoveGroupMember()
        {
            var recipientIDs = PromptUserList();

            string error;
            if (_testClient.Execute(new RemoveGroupMember(_currentGroupID, recipientIDs.ToArray()), out error))
            {
                Console.WriteLine("Successfully removed users.");
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void GroupMessage()
        {
            if (_currentGroupID == Guid.Empty)
            {
                Console.WriteLine("You must enter a group before sending it a message!");
                return;
            }

            Console.Write("Message: ");
            var message = Console.ReadLine();

            var found = _testClient.GroupCache.Groups.FirstOrDefault(g => g.GetGroup(_currentGroupID) != null);

            if (found == null)
            {
                Console.WriteLine("Unable to find the current group!");
                return;
            }

            var allTerms = found.Members.Select(m => new {Term = m.Value.Username, m.Value.UserID, m.Value.Username})
                .Concat(_testClient.FriendCache.Friends.Where(f => f.Nickname != null && found.Members.ContainsKey(f.UserID)).Select(f => new {Term = f.Nickname, f.UserID, f.Username}));

            var termMapping = new Dictionary<string, Tuple<int, string>>();
            foreach (var term in allTerms)
            {
                if (termMapping.ContainsKey(term.Term))
                {
                    continue;
                }

                termMapping.Add(term.Term, new Tuple<int, string>(term.UserID, term.Username));
            }

            string error;
            _testClient.Execute(new MessageGroup(_currentGroupID, MentionsHelper.ResolveMentionsForSend(message, termMapping)), out error);
        }

        private static void RenameGroup()
        {
            Console.Write("New Title: ");
            var title = Console.ReadLine();

            string error;
            if (_testClient.Execute(new ChangeGroupInfo(_currentGroupID) {Title = title}, out error))
            {
                Console.WriteLine("Changed Group {0}'s name to {1}", _currentGroupID, title);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void ToggleGroupFavorite()
        {
            Console.Write("Enter True/False:");
            var favorite = bool.Parse(Console.ReadLine());

            string error;
            if (_testClient.Execute(new ToggleFavoriteGroup(_currentGroupID, favorite), out error))
            {
                Console.WriteLine("Successfully toggled group {0} to favorite={1}", _currentGroupID, favorite);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void MuteGroup()
        {
            string error;
            if (_testClient.Execute(new ChangeGroupNotificationPreferences(_currentGroupID, NotificationPreference.Disabled), out error))
            {
                Console.WriteLine("Successfully muted {0}", _currentGroupID);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void UnmuteGroup()
        {
            string error;
            if (_testClient.Execute(new ChangeGroupNotificationPreferences(_currentGroupID, NotificationPreference.Enabled), out error))
            {
                Console.WriteLine("Successfully unmuted {0}", _currentGroupID);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void AddGroupMembers()
        {
            var recipientIDs = PromptUserList();

            string error;
            if (_testClient.Execute(new AddGroupMember(_currentGroupID, recipientIDs.ToArray()), out error))
            {
                Console.WriteLine("Successfully added users.");
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void CreateGroupInvite()
        {
            Console.Write("How long would you like this invite to be valid (Enter a number of hours or 0 for no expiration)? ");
            var hours = int.Parse(Console.ReadLine());

            bool autoRemoveGuests = false;
            if (hours > 0)
            {
                Console.Write("When this invite expires, should we automatically remove any guests that used it (Y|N)? ");
                autoRemoveGuests = Console.ReadLine().ToLower() == "y";
            }

            TimeSpan? lifespan = null;

            if (hours > 0)
            {
                lifespan = TimeSpan.FromHours(hours);
            }

            string error;
            string inviteCode;
            if (_testClient.Execute(new CreateGroupInvite(_currentGroupID){AutoRemoveGuests = autoRemoveGuests, LifeTime = lifespan}, out inviteCode, out error))
            {
                Console.WriteLine("Successfully created invite {0} for group {1}", inviteCode, _currentGroupID);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void ProcessGroupInvite()
        {
            Console.Write("Enter an invite code: ");
            var code = Console.ReadLine();

            string error;
            if (_testClient.Execute(new ClaimGroupInvite(code), out error))
            {
                Console.WriteLine("Successfully claimed invite code {0}", code);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void DeleteGroupInvite()
        {
            Console.Write("Enter an invite code: ");
            var code = Console.ReadLine();

            Console.Write("Would you like to also remove any guests that used this code (Y|N)? ");
            var removeGuests = Console.ReadLine().ToLower() == "y";

            string error;
            if (_testClient.Execute(new DeleteGroupInvite(code) {RemoveGuests = removeGuests}, out error))
            {
                Console.WriteLine("Successfully deleted group invite {0}", code);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void ChangeGroupMemberRole()
        {
            Console.WriteLine("Not Implemented as of refactor!");
            Console.ReadLine();

            //var recipientIDs = PromptUserList(1);

            ////var role = PromptRole("Enter a new group role for this user.");

            //string error;
            //if (_testClient.Execute(new ChangeGroupMemberRole(_currentGroupID, recipientIDs.First()), out error))
            //{
            //    Console.WriteLine("Successfully changed {0} member role to {1} in group {2}.", recipientIDs.First(), "DEPRECATED", _currentGroupID);
            //}
            //else
            //{
            //    Console.WriteLine(error);
            //}
        }

        private static void CallGroup()
        {

            Console.WriteLine("Creating or joining a group call...");

            string error;
            string inviteCode;
            if (_testClient.Execute(new CallGroup(_currentGroupID) {SendInvitation = true}, out inviteCode, out error))
            {
                Console.WriteLine("Group voice call succeeded! " + inviteCode);
            }
            else
            {
                Console.WriteLine("Group voice call failed :( - {0}", error);
            }

            Console.ReadKey();
        }

        private static void CallFriend()
        {


            Console.WriteLine("Friend User ID: ");
            var input = Console.ReadLine();

            int userID;
            if (!int.TryParse(input, out userID))
            {
                Console.WriteLine("Invalid friend ID.");
                Console.ReadKey();
                return;
            }

            string error;
            string inviteCode;
            if (_testClient.Execute(new CallFriend(userID) {SendInvitation = true}, out inviteCode, out error))
            {
                Console.WriteLine("Friend call succeeded! " + inviteCode);
            }
            else
            {
                Console.WriteLine("Friend call failed :( - {0}", error);
            }

            Console.ReadKey();
        }

        private static void EnterGroup()
        {
            var groupID = PromptGroup();

            // Try to get this from the group cache
            GroupCache found = _testClient.GroupCache.Groups.FirstOrDefault(g => g.GetGroup(groupID) != null);
            if (found == null)
            {
                Console.WriteLine("Unknown group!");
                return;
            }

            _currentGroupID = groupID;
        }

        private static void ChangeStatus()
        {

            var enumOptions = Enum.GetValues(typeof (UserConnectionStatus));
            foreach (var option in enumOptions)
            {
                Console.WriteLine((int) option + ". " + option.ToString());
            }


            var newStatus = (UserConnectionStatus) int.Parse(Console.ReadLine());

            Console.Write("Enter an optional status message: ");
            var statusMessage = Console.ReadLine();

            string error;
            _testClient.Execute(new ChangeStatus(newStatus) {StatusMessage = statusMessage}, out error);
        }


        private static void ChangeGameStatus()
        {
            Console.Write("Enter a game ID: ");
            var gameID = int.Parse(Console.ReadLine());

            Console.Write("Enter a game status message: ");
            string gameStatus = Console.ReadLine();

            string error;
            _testClient.Execute(new ChangeGameStatus(gameID) {StatusMessage = gameStatus, IsRunning = gameID > 0, GameState = 1}, out error);
        }

        private static void UpdateProfile()
        {
            Console.Write("Enter an avatar URL: ");
            string avatarUrl = Console.ReadLine();

            string error;
            _testClient.Execute(new ChangeProfile {AvatarUrl = avatarUrl}, out error);
        }

        private static void SendFriendRequest()
        {

            Console.Write("Enter a friend ID: ");
            var friendID = int.Parse(Console.ReadLine());
            Console.Write("Enter the name of your friend: ");
            string knownIdentity = Console.ReadLine();

            string error;
            RequestFriendResponseType responseType;
            if (_testClient.Execute(new RequestFriend(friendID, knownIdentity), out responseType, out error))
            {
                Console.WriteLine("Successffully requested friendship with {0} ({1})", knownIdentity, friendID);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void Disconnect()
        {
            string error;
            _testClient.Logout(out error);
            _currentGroupID = Guid.Empty;
            _currentVoiceAccessToken = null;
            _currentVoiceInviteGroup = null;
            _currentVoiceInviteUrl = null;
            _currentVoiceInviteUserID = null;
        }

        private static void Login()
        {
            // Need to disconnect here!
            if (_testClient.IsLoggedIn)
            {
                Disconnect();
            }

            Console.Write("Enter your username: ");
            string username = Console.ReadLine();
            Console.Write("Enter your password: ");
            var password = GetPassword();

            var credentials = new NetworkCredential(username, password);

            Console.WriteLine();
            Console.Write("Select a Device Platform (0=Windows, 1=iOS, 2=Android): ");
            string platform = Console.ReadLine();

            Device device;
            int platformOrdinal;
            if (!int.TryParse(platform, out platformOrdinal) || platformOrdinal < 1 || platformOrdinal > 2)
            {
                platformOrdinal = 0;
            }

            if (platformOrdinal == 1)
            {
                device = Device.iOS;
            }
            else if (platformOrdinal == 2)
            {
                device = Device.Android;
            }
            else
            {
                device = Device.Windows;
            }

            string errorMessage;
            if (!_testClient.Login(new LoginInfo(credentials) {Device = device}, out errorMessage))
            {
                Console.WriteLine("Failed to login. " + errorMessage);
            }

            _testClient.Events.Register<NotificationsDisconnected>(_client_Disconnected);
        }

        private static void UnregisterEndpoint()
        {
            string errorMessage;
            _testClient.Execute(new UnregisterEndpoint(), out errorMessage);
            Disconnect();
        }

        private static void ConfirmFriendRequest()
        {
            Console.Write("Enter a friend ID: ");
            var userID = int.Parse(Console.ReadLine());

            string error;
            if (_testClient.Execute(new ConfirmFriend(userID), out error))
            {
                Console.WriteLine("Confirmed friendship with {0}", userID);
            }
            else
            {
                Console.WriteLine(error);
            }
        }

        private static void AcceptVoiceInvite()
        {
            if (_currentVoiceInviteGroup.HasValue || _currentVoiceInviteUserID.HasValue)
            {
                Console.WriteLine("Accepting Call {0} from {1}", _currentVoiceInviteUrl,
                    _currentVoiceInviteGroup.HasValue ? "Group " + _currentVoiceInviteGroup.Value : _currentVoiceInviteUserID.Value.ToString());

                string error;
                if (_currentVoiceInviteGroup.HasValue && !_testClient.Execute(new AcceptGroupCall(_currentVoiceInviteGroup.Value,_currentVoiceInviteUrl), out error))
                {
                    Console.WriteLine("Failed to join the group call. {0}", error);
                }
                else if(_currentVoiceInviteUserID.HasValue && !_testClient.Execute(new AcceptFriendCall(_currentVoiceInviteUserID.Value,_currentVoiceInviteUrl,_currentVoiceAccessToken.Value), out error))
                {
                    Console.WriteLine("Failed to join the friend call. {0}", error);
                }
            }
        }

        private static void DeclineVoiceInvite()
        {
            string error;
            if (_currentVoiceInviteGroup.HasValue)
            {
                Console.WriteLine("Declining Call {0} from {1}", _currentVoiceInviteUrl, "Group " + _currentVoiceInviteGroup.Value);
                if (!_testClient.Execute(new DeclineGroupCall(_currentVoiceInviteGroup.Value, _currentVoiceInviteUrl), out error))
                {
                    Console.WriteLine(error);
                }
            }
            else if (_currentVoiceInviteUserID.HasValue)
            {
                Console.WriteLine("Declining Call {0} from {1}", _currentVoiceInviteUrl, _currentVoiceInviteUserID.Value);
                if (!_testClient.Execute(new DeclineFriendCall(_currentVoiceInviteUserID.Value, _currentVoiceInviteUrl), out error))
                {
                    Console.WriteLine(error);
                }
            }
            else
            {
                return;
            }

            _currentVoiceInviteGroup = null;
            _currentVoiceInviteUserID = null;
            _currentVoiceInviteUrl = null;
            _currentVoiceAccessToken = null;
        }

        private static void FavoriteFriend()
        {
            Console.Write("Enter a friend ID to favorite: ");
            var userID = int.Parse(Console.ReadLine());

            string error;
            if (!_testClient.Execute(new ToggleFavoriteFriend(userID, true), out error))
            {
                Console.WriteLine(error);
            }
        }

        private static void RenameFriend()
        {
            Console.Write("Enter a friend ID to rename: ");
            var userID = int.Parse(Console.ReadLine());
            Console.Write("Enter the new name: ");
            var nickname = Console.ReadLine();

            string error;
            if (!_testClient.Execute(new RenameFriend(userID, nickname), out error))
            {
                Console.WriteLine(error);
            }
        }

        private static void DeclineFriendRequest()
        {
            Console.Write("Enter a friend ID to decline: ");
            var userID = int.Parse(Console.ReadLine());

            Console.Write("Do you want to block this user from sending you requests in the future [Y|N]: ");
            var blockFuture = Console.ReadKey().Key == ConsoleKey.Y;


            string error;
            if (!_testClient.Execute(new DeclineFriendRequest(userID) {BlockFutureRequests = blockFuture}, out error))
            {
                Console.WriteLine(error);
            }
        }

        private static void RemoveFriend()
        {
            Console.Write("Enter a friend ID to remove: ");
            var userID = int.Parse(Console.ReadLine());

            string error;
            _testClient.Execute(new RemoveFriend(userID), out error);

        }


        private static void UnblockFriend()
        {
            Console.Write("Enter a friend ID to unblock: ");
            var userID = int.Parse(Console.ReadLine());

            string error;
            _testClient.Execute(new UnblockFriend(userID), out error);
        }

        private static void Search()
        {
            while (true)
            {
                Console.Write("Username, character or complete email address (X to exit): ");

                var query = Console.ReadLine();
                if (query.ToLower().Equals("x"))
                {
                    return;
                }

                Console.Clear();

                string error;
                FriendSearchResponse resp;

                if (!_testClient.Execute(new FindFriends(query), out resp, out error))
                {
                    Console.WriteLine(error);
                    Console.WriteLine("Press any key to continue...");
                    Console.ReadKey(true);
                    continue;
                }

                Console.WriteLine("Search completed in " + resp.ElapsedMilliseconds + " ms");
                Console.WriteLine();

                if (resp.EmailMatches.Any())
                {
                    Console.WriteLine("Email Matches");
                    Console.WriteLine();
                    foreach (var m in resp.EmailMatches)
                    {
                        Console.WriteLine("- " + m.Value.EmailAddress);
                    }
                }


                if (resp.UserMatches.Any())
                {
                    Console.WriteLine("User Matches");
                    Console.WriteLine();

                    foreach (var m in resp.UserMatches)
                    {
                        Console.WriteLine("- " + m.Value.Username + " (" + m.Value.FriendCount + " friends)");
                    }
                }

                if (resp.CharacterMatches.Any())
                {
                    Console.WriteLine("Characters");
                    Console.WriteLine();
                    foreach (var m in resp.CharacterMatches)
                    {
                        Console.WriteLine("- " + m.Value.CharacterName + "@" + m.Value.ServerName + " - " + m.Value.ServerRegion);
                    }


                }

                Console.WriteLine();
            }
        }


        private static void InstantMessage()
        {
            Console.Write("Enter a friend ID: ");
            var userID = int.Parse(Console.ReadLine());
            Console.Write("Message: ");
            var message = Console.ReadLine();

            string error;
            _testClient.Execute(new MessageFriend(userID, message), out error);
        }

        private static void SyncOfflineMessages()
        {
            Console.WriteLine("How many days do you want to sync?");
            int days;
            if (int.TryParse(Console.ReadLine(), out days))
            {
                Console.Clear();
                Console.WriteLine("Sending offline message request. Press any key to continue...");
                Console.WriteLine();

                string error;
                _testClient.Execute(new SyncOfflineMessages(DateTime.UtcNow.AddDays(-days)), out error);
                Console.ReadKey();
            }
            else
            {
                Console.WriteLine("Invalid number of days, Press any key to continue.");
                Console.ReadKey();
            }

        }

        private static void HangUp()
        {
            if (_currentCallInviteCode==null)
            {
                Console.WriteLine("You are not in a call!");
            }
            else
            {
                string error;
                if (_testClient.Execute(new HangUp(), out error))
                {
                    Console.WriteLine("Hung up call.");
                }
                else
                {
                    Console.WriteLine("Failed to hang up. {0}", error);
                }
            }
            Console.ReadKey();
        }

        private static void GetProfile()
        {
            Console.Write("Enter a user ID: ");
            var userID = int.Parse(Console.ReadLine());

            UserProfileResponse resp;
            string error;
            if (_testClient.Execute(new GetProfile(userID), out resp, out error))
            {
                Console.WriteLine("Found user profile:");
                Console.Write(resp.Username);
                Console.WriteLine(resp.Name);

                foreach (var f in resp.MutualFriends)
                {
                    Console.WriteLine("Mutual Friend: " + f.UserID + " (" + f.MutualUserIDs.Length + ")");
                }

                Console.ReadLine();
            }
        }

        private static void FriendsNotificationClientOnOfflineMessageNotification(OfflineMessagesSynced value)
        {
            foreach (var groupConvo in value.GroupMessages)
            {
                Console.WriteLine("Received offline messages for group " + groupConvo.GroupID + ". Total Messages: " + groupConvo.Count);
                foreach (var message in groupConvo.Messages)
                {
                    ProcessGroupMessage((GroupMessageReceived)message);
                }
                Console.WriteLine("-----------------------------------------------------------------------------------");
                Console.WriteLine();
            }

            foreach (var friendConvo in value.InstantMessages)
            {
                Console.WriteLine("Received offline messages from friend " + friendConvo.FriendID + ". Total Messages: " + friendConvo.Count);
                foreach (var message in friendConvo.Messages)
                {
                    ProcessInstantMessage((FriendMessageReceived)message);
                }
                Console.WriteLine("-----------------------------------------------------------------------------------");
                Console.WriteLine();

            }
        }

        private static void FriendsNotificationClientOnGroupChangeNotification(GroupChanged value)
        {
            var notificationGroup = value.Group;

            GroupCache cache = _testClient.GroupCache.Groups.FirstOrDefault(g => g.GetGroup(notificationGroup.GroupID) != null);

            if (cache==null)
            {
                // Add it!
                if (notificationGroup.GroupID == notificationGroup.RootGroupID)
                {
                    Console.WriteLine("New group found: {0} ({1})", notificationGroup.GroupTitle, notificationGroup.GroupID);
                }
                else
                {
                    Console.WriteLine("Received a notification for an unknown parented group!");
                }
                return;
            }

            var sender = cache.GetMember(value.SenderID);
            var senderName = sender != null ? sender.Username : "User#" + value.SenderID;

            switch (value.ChangeType)
            {
                case GroupChangeType.CreateGroup:
                    Console.WriteLine(senderName + " created group " + notificationGroup.GroupTitle);
                    break;

                case GroupChangeType.RemoveGroup:
                    Console.WriteLine(senderName + " removed group " + notificationGroup.GroupTitle);
                    break;

                case GroupChangeType.AddUsers:

                    foreach (var newUser in value.Members)
                    {
                        cache.AddMember(newUser);
                        Console.WriteLine(newUser.Username + " has joined " + cache.RootGroup.GroupTitle);
                    }

                    break;

                case GroupChangeType.RemoveUsers:

                    if (value.Members.Any(p => p.UserID == _testClient.UserID))
                    {
                        Console.WriteLine("You have been removed from " + cache.RootGroup.GroupTitle);
                    }
                    else
                    {
                        foreach (var removedUser in value.Members)
                        {
                            Console.WriteLine(removedUser.Username + " has left " + cache.RootGroup.GroupTitle);
                        }
                    }

                    break;

                case GroupChangeType.UpdateUsers:

                    foreach (var u in value.Members)
                    {
                        Console.WriteLine(senderName + " has updated " + u.Username + " (" + u.BestRole + ") in " + cache.RootGroup.GroupTitle);
                    }

                    break;

                case GroupChangeType.ChangeInfo:
                {
                    var updatedGroup = cache.AddOrUpdateGroup(notificationGroup);
                    Console.WriteLine("{0} changed group title to {1} and avatar url to {2}", senderName, updatedGroup.GroupTitle, updatedGroup.GroupAvatar);
                    break;
                }

                case GroupChangeType.GroupReorganized:
                {
                    Console.WriteLine("Group '" + notificationGroup.GroupTitle + "' has been reorganized!");
                    foreach (var group in value.ChildGroups)
                    {
                        Console.WriteLine("Child group '" + group.GroupTitle + "' has been reorganized!");
                    }

                    break;
                }
                case GroupChangeType.VoiceSessionStarted:
                {


                    break;
                }
                    break;
            }

        }

        private static void FriendsNotificationClient_GroupMessageNotification(GroupMessageReceived value)
        {
            ProcessGroupMessage(value);
        }

        private static void ProcessGroupMessage(GroupMessageReceived value)
        {
            // Try to get the group from the group membership dictionary
            GroupCache cache = _testClient.GroupCache.Groups.FirstOrDefault(g => g.GetGroup(value.GroupID) != null);
            if (cache != null)
            {
                GroupMemberNotification sender;
                var senderName = "User " + value.SenderID;
                if (cache.Members.TryGetValue(value.SenderID, out sender))
                {
                    senderName = sender.Username;
                }

                Console.WriteLine(senderName + " says to " + cache.RootGroup.GroupTitle + ": " +
                                  MentionsHelper.ResolveMentionsForDisplay(value.Message, _testClient.FriendCache.Friends.Where(f => f.Nickname != null).ToDictionary(f => f.UserID, f => f.Nickname)));

                // Notify that the group message has been read
                string error;
                _testClient.Execute(new SendConversationRead(value.GroupID), out error);
            }
        }

        private static void FriendsNotificationClient_VoiceDeclineNotification(FriendCallDeclined value)
        {
            Console.WriteLine("User " + value.SenderID + " has declined your voice invitation");
        }

        private static void FriendsNotificationClient_VoiceDeclineNotification(GroupCallDeclined value)
        {
            Console.WriteLine("User " + value.SenderID + " has declined your voice invitation");
        }

        private static void FriendsNotificationClient_VoiceInvitationNotification(FriendCallInvitationReceived value)
        {
            _currentVoiceInviteUserID = value.SenderID;
            _currentVoiceInviteUrl = value.InviteUrl;
            _currentVoiceInviteGroup = null;
            _currentVoiceAccessToken = value.AccessToken;
            Console.WriteLine("Received a voice invite from user " + _currentVoiceInviteUserID + ": " + _currentVoiceInviteUrl);
        }

        private static void FriendsNotificationClient_GroupVoiceInvitationNotification(GroupCallInvitationReceived value)
        {
            _currentVoiceInviteUrl = value.InviteUrl;
            _currentVoiceInviteUserID = null;
            _currentVoiceInviteGroup = value.GroupID;
            _currentVoiceAccessToken = null;
            Console.WriteLine("Received a voice invite for group " + _currentVoiceInviteGroup + " from user " + value.RequestorID + ": " + _currentVoiceInviteUrl);
        }

        private static void _client_Disconnected(NotificationsDisconnected value)
        {
            Console.WriteLine("Client disconnected!");
        }

        private static void Instance_UserChangeNotification(UserChanged e)
        {
            Console.WriteLine("Your user info changed! Avatar is: " + e.User.AvatarUrl);

        }

        private static void _notificationClient_FriendshipRemovedNotification(FriendshipRemoved value)
        {

            Console.WriteLine("Your friendship with " + value.FriendID + " has beeen dissolved!");
        }

        private static void _notificationClient_FriendshipChangeNotification(FriendshipChanged value)
        {
            var friendship = value.Friendship;
            Console.WriteLine("Your friendship with " + (friendship.OtherUsername ?? friendship.OtherUserNickname) + " has updated.");
            Console.WriteLine("Status: " + friendship.Status
                              + Environment.NewLine
                              + "Avatar: " + friendship.OtherUserAvatarUrl
                              + Environment.NewLine
                              + "Game: " + friendship.OtherUserGameID + ", " + friendship.OtherUserGameState);
        }

        private static void _notificationClient_InstantMessageResponse(MessageSent e)
        {
            if (e.Status != DeliveryStatus.Successful)
            {
                Console.WriteLine("Unable to send message: " + e.Status);
            }
        }

        private static void _notificationClient_InstantMessageNotification(FriendMessageReceived value)
        {
            ProcessInstantMessage(value);
        }

        private static void ProcessInstantMessage(FriendMessageReceived value)
        {
            Console.WriteLine(value.Timestamp.ToLocalTime().ToShortTimeString() + " - " + value.SenderID + ": " + value.Message);

            // Notify that the IM has been read
            string error;
            _testClient.Execute(new SendConversationRead(_testClient.UserID == value.SenderID ? value.RecipientID : value.SenderID), out error);
        }
    }
}
