﻿using System;
using System.Threading;
using System.Threading.Tasks;
using System.Windows.Input;
using Curse.Friends.Client.FriendsService;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.UI.ViewModels;

namespace Curse.FriendsService.Tester.UI.Notifications
{
    class CallNotification : BaseNotification
    {
        private readonly TestClient _client;
        private readonly CancellationTokenSource _cts = new CancellationTokenSource();

        public CallNotification(TestClient client, string inviteCode, long? accessToken, int senderID, GroupViewModel group=null)
        {
            _client = client;
            AcceptCommand = new RelayCommand(Accept);
            DeclineCommand = new RelayCommand(Decline);
            Timestamp = DateTime.UtcNow;
            InviteCode = inviteCode;
            AccessToken = accessToken;
            SenderID = senderID;
            SenderName = GetSenderName(client, senderID);
            if (group != null)
            {
                GroupID = group.GroupID;
                GroupName = group.Title;
            }

            Task.Delay(TimeSpan.FromSeconds(30), _cts.Token).ContinueWith(_ =>
            {
                Response = Response ?? "Did not respond.";
            }, _cts.Token, TaskContinuationOptions.NotOnCanceled, TaskScheduler.Default);
        }

        private static string GetSenderName(TestClient client, int senderID)
        {
            string error;
            UserProfileResponse profile;
            if (client.Execute(new GetProfile(senderID), out profile, out error))
            {
                return profile.Username;
            }
            return "Unknown - " + error;
        }

        public ICommand AcceptCommand { get; private set; }

        public ICommand DeclineCommand { get; private set; }

        private void Decline()
        {
            string error;
            if (GroupID.HasValue)
            {
                Response = _client.Execute(new DeclineGroupCall(GroupID.Value, InviteCode), out error) ? "Declined" : error;
            }
            else
            {
                Response = _client.Execute(new DeclineFriendCall(SenderID, InviteCode), out error) ? "Declined" : error;
            }
            _cts.Cancel();
        }

        private void Accept()
        {
            Response = "Accepted";
            string error;
            if (GroupID.HasValue)
            {
                if (!_client.Execute(new AcceptGroupCall(GroupID.Value, InviteCode), out error))
                {
                    Response = error;
                }
            }
            else
            {
                if (!_client.Execute(new AcceptFriendCall(SenderID, InviteCode, AccessToken.Value), out error))
                {
                    Response = error;
                }
            }
            _cts.Cancel();
        }

        public string InviteCode { get; private set; }
        public long? AccessToken { get; private set; }
        public int SenderID { get; private set; }
        public string SenderName { get; private set; }
        public Guid? GroupID { get; private set; }
        public string GroupName { get; private set; }

        private string _response;
        public string Response
        {
            get { return _response; }
            set
            {
                if (_response != value)
                {
                    _response = value;
                    OnPropertyChanged();
                }
            }
        }
    }
}
