﻿using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.UI.Messaging;

namespace Curse.FriendsService.Tester.UI.Options
{

    internal class FindFriendCommand : BaseCommandOption
    {
        public FindFriendCommand(IMessenger messenger) : base(messenger)
        {
            Results = new ObservableCollection<FindResult>();
        }

        public override string Name
        {
            get { return "Find Friends"; }
        }

        public override void Execute(TestClient client)
        {
            Results.Clear();

            string error;
            FriendSearchResponse result;
            if (client.Execute(new FindFriends(Query), out result, out error))
            {
                var results = GetFindResults(result);
                foreach (var findResult in results)
                {
                    AddResult(findResult);
                }
            }
            else
            {
                SendNotificationMessage(error);
            }
        }

        private void AddResult(FindResult result)
        {
            var idx = Results.SkipWhile((r, i) => r.Score > result.Score).Select((r, i) => i).FirstOrDefault();
            Results.Insert(idx, result);
        }

        public ObservableCollection<FindResult> Results { get; private set; }

        public string Query { get; set; }

        private static List<FindResult> GetFindResults(FriendSearchResponse response)
        {
            var findResults = new List<FindResult>();

            if (response.CharacterMatches != null)
            {
                findResults.AddRange(response.CharacterMatches.Select(m => new FindResult
                {
                    Type = FindResultType.Character,
                    Text = m.Value.CharacterName,
                    Score = m.Score,
                    UserID = m.Value.UserID
                }));
            }

            if (response.EmailMatches != null)
            {
                findResults.AddRange(response.EmailMatches.Select(m => new FindResult
                {
                    Type = FindResultType.Email,
                    Text = m.Value.EmailAddress,
                    Score = m.Score,
                    UserID = m.Value.UserID
                }));
            }

            if (response.PlatformMatches != null)
            {
                findResults.AddRange(response.PlatformMatches.Select(m => new FindResult
                {
                    Type = FindResultType.Platform,
                    Text = m.Value.Platform.ToFriendlyString(),
                    Score = m.Score,
                    UserID = m.Value.UserID
                }));
            }

            if (response.UserMatches != null)
            {
                findResults.AddRange(response.UserMatches.Select(m => new FindResult
                {
                    Type = FindResultType.User,
                    Text = m.Value.Username,
                    Score = m.Score,
                    UserID = m.Value.UserID
                }));
            }
            return findResults;
        }
    }

    public class FindResult
    {
        public FindResultType Type { get; set; }

        public string Text { get; set; }

        public double Score { get; set; }

        public int UserID { get; set; }
    }

    public enum FindResultType
    {
        Character,
        Email,
        Platform,
        User
    }

}
