﻿using System.Collections.ObjectModel;
using System.Linq;
using System.Windows.Input;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.UI.Messaging;
using Curse.FriendsService.Tester.UI.ViewModels;

namespace Curse.FriendsService.Tester.UI.Options
{
    class RemoveGroupMemberCommand : BaseCommandOption
    {
        public RemoveGroupMemberCommand(IMessenger messenger, GroupsList groupsList) : base(messenger)
        {
            AddUserToRemoveCommand = new RelayCommand<GroupMemberViewModel>(AddUserToRemove);
            RemoveUserToRemoveCommand = new RelayCommand<GroupMemberViewModel>(RemoveUserToRemove);

            Groups = groupsList.Groups;
            Group = Groups.FirstOrDefault();
            MembersToRemove = new DispatchedObservableCollection<GroupMemberViewModel>();
        }

        public override string Name
        {
            get { return "Remove Group Member"; }
        }

        public override void Execute(TestClient client)
        {
            if (!MembersToRemove.Any())
            {
                SendNotificationMessage("Nobody is selected for removal from the group.");
            }

            string error;
            if (client.Execute(new RemoveGroupMember(Group.GroupID, MembersToRemove.Select(m => m.UserID).ToArray()), out error))
            {
                SendNotificationMessage("Group {0}: Successfully removed members from group {1}.", Group.Title, Group.GroupID);
            }
            else
            {
                SendNotificationMessage("Group {0}: {1}", Group.Title, error);
            }
        }

        public ObservableCollection<GroupViewModel> Groups { get; private set; }

        private GroupViewModel _group;
        public GroupViewModel Group
        {
            get { return _group; }
            set
            {
                if (_group != value)
                {
                    _group = value;

                    if (_group != null)
                    {
                        _group.RefreshGroupDetails();
                    }

                    OnPropertyChanged();

                    if (_group != null)
                    {
                        CurrentGroupMembers = _group.Members;
                    }
                }
            }
        }

        public ObservableCollection<GroupMemberViewModel> MembersToRemove { get; private set; }

        private ObservableCollection<GroupMemberViewModel> _currentGroupMembers;
        public ObservableCollection<GroupMemberViewModel> CurrentGroupMembers
        {
            get { return _currentGroupMembers; }
            private set
            {
                if (_currentGroupMembers != value)
                {
                    _currentGroupMembers = value;
                    OnPropertyChanged();
                }
            }
        }

        public ICommand AddUserToRemoveCommand { get; private set; }
        public ICommand RemoveUserToRemoveCommand { get; private set; }

        private void RemoveUserToRemove(GroupMemberViewModel user)
        {
            MembersToRemove.Remove(user);
        }

        private void AddUserToRemove(GroupMemberViewModel user)
        {
            if (user == null)
            {
                return;
            }

            MembersToRemove.Add(user);
        }
    }
}
