﻿using System.Collections.ObjectModel;
using System.Windows.Input;
using Curse.Friends.Enums;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.People;

namespace Curse.FriendsService.Tester.UI.ViewModels
{
    internal class FriendViewModel : UserViewModel
    {
        private readonly Friend _friend;

        public FriendViewModel(TestClient client, Friend friend) : base(client,friend)
        {
            _friend = friend;

            ConfirmFriendshipCommand = new RelayCommand(ConfirmFriendship);
            DeclineFriendshipCommand = new RelayCommand(() => DeclineFriendship(false));
            BlockFriendshipCommand = new RelayCommand(() => DeclineFriendship(true));
            MutualFriends = new DispatchedObservableCollection<MutualFriendViewModel>();

            friend.Updated += FriendUpdated;
        }

        void FriendUpdated(object sender, FriendEventArgs args)
        {
            // Update
            ConnectionStatus = args.AffectedFriend.ConnectionStatus;
            Nickname = args.AffectedFriend.Nickname;
            FriendshipStatus = args.AffectedFriend.FriendshipStatus;
            Favorite = args.AffectedFriend.Favorite;
            StatusMessage = args.AffectedFriend.StatusMessage;
            GameID = args.AffectedFriend.GameID;
            GameStatusMessage = args.AffectedFriend.GameStatusMessage;

            OnPropertyChanged(string.Empty);
        }

        public ICommand ConfirmFriendshipCommand { get; private set; }
        public ICommand DeclineFriendshipCommand { get; private set; }
        public ICommand BlockFriendshipCommand { get; private set; }

        private void ConfirmFriendship()
        {
            string error;
            Client.Execute(new AcceptFriendRequest(UserID), out error);
        }

        private void DeclineFriendship(bool block)
        {
            string error;
            Client.Execute(new DeclineFriendRequest(UserID) {BlockFutureRequests = block}, out error);
        }

        public string Nickname
        {
            get { return _friend.Nickname; }
            set
            {
                if (Nickname != value)
                {
                    _friend.Nickname = value;
                    OnPropertyChanged();
                }
            }
        }

        private FriendshipStatus _friendshipStatus;
        public FriendshipStatus FriendshipStatus
        {
            get { return _friend.FriendshipStatus; }
            set
            {
                if (FriendshipStatus != value)
                {
                    _friend.FriendshipStatus = value;
                    OnPropertyChanged();
                }
            }
        }

        public bool Favorite
        {
            get { return _friend.Favorite; }
            set
            {
                if (Favorite != value)
                {
                    _friend.Favorite = value;
                    OnPropertyChanged();
                }
            }
        }

        public ObservableCollection<MutualFriendViewModel> MutualFriends { get; private set; }

        protected bool Equals(FriendViewModel other)
        {
            return UserID == other.UserID;
        }

        protected override void UpdateProfileFields(User profile)
        {
            var friend = (Friend) profile;
            MutualFriends.Clear();
            foreach (var mutualFriend in friend.MutualFriends)
            {
                MutualFriends.Add(new MutualFriendViewModel
                {
                    UserID = mutualFriend.UserID,
                    NumberOfMutualFriends = mutualFriend.MutualFriendCount
                });
            }
        }

        public override bool Equals(object obj)
        {
            if (ReferenceEquals(null, obj)) return false;
            if (ReferenceEquals(this, obj)) return true;
            var other = obj as FriendViewModel;
            return other != null && Equals(other);
        }

        public override int GetHashCode()
        {
            return UserID;
        }
    }
}
