﻿using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Groups;
using Curse.FriendsService.Tester.UI.Messaging;

namespace Curse.FriendsService.Tester.UI.ViewModels
{
    internal class GroupViewModel : BaseViewModel
    {
        private readonly TestClient _client;
        private readonly IMessenger _messenger;
        private readonly FriendsList _friendsList;
        private GroupCache _group;

        public GroupViewModel(GroupCache group, TestClient client, IMessenger messenger, FriendsList friendsList)
        {
            _client = client;
            _group = group;
            _messenger = messenger;
            _friendsList = friendsList;

            Members = new DispatchedObservableCollection<GroupMemberViewModel>();
            AllGroupInvites = new DispatchedObservableCollection<GroupInviteViewModel>();

            if (group != null)
            {
                group.MembersAdded += GroupMembersAdded;
                group.MembersRemoved += GroupMembersRemoved;
                group.MembersUpdated += GroupMembersUpdated;
                group.InfoChanged += GroupInfoChanged;
            }
        }

        private void GroupMembersAdded(object sender, GroupMemberEventArgs e)
        {
            AddUsers(e.AffectedMembers);
        }

        void GroupMembersRemoved(object sender, GroupMemberEventArgs e)
        {
            RemoveUsers(e.AffectedMembers);
        }

        void GroupMembersUpdated(object sender, GroupMemberEventArgs e)
        {
            UpdateUsers(e.AffectedMembers);
        }

        private void GroupInfoChanged(object sender, GroupEventArgs e)
        {
            _group = e.AffectedGroup;
            OnPropertyChanged(string.Empty);
        }


        private Guid _groupID;

        public Guid GroupID
        {
            get { return _group == null ? _groupID : _group.RootGroup.GroupID; }
            set
            {
                if (_group != null)
                {
                    throw new Exception("Can't manually change the Group ID when the VM is backed by a GroupCache");
                }
                if (_groupID != value)
                {
                    _groupID = value;
                    OnPropertyChanged();
                }
            }
        }

        private string _title;
        public string Title
        {
            get { return _group == null ? _title : _group.RootGroup.GroupTitle; }
            set
            {
                if (_group != null)
                {
                    throw new Exception("Can't manually change the Group Title when the VM is backed by a GroupCache");
                }
                if (_title != value)
                {
                    _title = value;
                    OnPropertyChanged();
                }
            }
        }

        public bool Muted
        {
            get { return _group != null && _group.RootGroup.NotificationPreference==NotificationPreference.Disabled; }
        }

        public bool Favorite
        {
            get { return _group != null && _group.RootGroup.IsFavorite; }
            set
            {
                if (_group.RootGroup.IsFavorite != value)
                {
                    _group.RootGroup.IsFavorite = value;
                    OnPropertyChanged();
                }
            }
        }

        public ObservableCollection<GroupMemberViewModel> Members { get; private set; }

        private void UpdateUsers(GroupMemberNotification[] members)
        {
            foreach (var member in members)
            {
                var vm = Members.FirstOrDefault(m => m.UserID == member.UserID);
                if (vm == null)
                {
                    continue;
                }
                vm.UpdateInfo(member);
            }
        }

        private void AddUsers(GroupMemberNotification[] members)
        {
            foreach (var member in members)
            {
                var vm = Members.FirstOrDefault(m => m.UserID == member.UserID);
                if (vm != null)
                {
                    continue;
                }
                Members.Add(new GroupMemberViewModel
                {
                    UserID = member.UserID,
                    Username = member.Username,
                });
            }
        }

        private void RemoveUsers(GroupMemberNotification[] members)
        {
            foreach (var member in members)
            {
                var vm = Members.FirstOrDefault(m => m.UserID == member.UserID);
                Members.Remove(vm);
            }
        }

        private void RemoveUsers(GroupMemberViewModel[] members)
        {
            foreach (var member in members)
            {
                Members.Remove(member);
            }
        }

        public void RefreshGroupDetails()
        {
            string error;
            GetGroupDetailsResponse response;
            if (_client.Execute(new GetGroupDetails(GroupID){IncludePermissions = true,IncludeRoleNames = true,IncludeDeletedChannels = true}, out response, out error))
            {
                MyActiveInvite = GetGroupInvites(false).FirstOrDefault();

                AllGroupInvites.Clear();
                foreach (var invite in GetGroupInvites(true))
                {
                    AllGroupInvites.Add(invite);
                }

                if (!AllGroupInvites.Any() && MyActiveInvite != null)
                {
                    AllGroupInvites.Add(MyActiveInvite);
                }

                RemoveUsers(Members.Where(vm => response.Members.All(m => m.UserID != vm.UserID)).ToArray());
                AddUsers(response.Members.Where(m => Members.All(vm => m.UserID != vm.UserID)).ToArray());
                UpdateUsers(response.Members.Where(m => Members.Any(vm => m.UserID == vm.UserID)).ToArray());
            }
            else
            {
                _messenger.Raise(new NotificationMessage {Message = string.Format("Group {0}: {1}", Title, error)});
            }
        }

        private IEnumerable<GroupInviteViewModel> GetGroupInvites(bool mineOnly)
        {
            string error;
            GroupInvitation[] invites;
            if (_client.Execute(new GetGroupInvites(GroupID) {MineOnly = mineOnly}, out invites, out error))
            {
                return invites.Select(i => new GroupInviteViewModel
                {
                    Invite = i.InviteCode,
                    AutoRemoveMembers = i.AutoRemoveMembers,
                    Created = i.DateCreated,
                    Expires = i.DateExpires,
                    Creator = i.CreatorID == _client.UserID
                        ? (UserViewModel)new CurrentUserViewModel(_client)
                        : _friendsList.Friends.FirstOrDefault(f => f.UserID == i.CreatorID)
                });
            }

            return new GroupInviteViewModel[0];
        } 

        public ObservableCollection<GroupInviteViewModel> AllGroupInvites { get; private set; }

        private GroupInviteViewModel _myActiveInvite;
        public GroupInviteViewModel MyActiveInvite
        {
            get { return _myActiveInvite; }
            private set
            {
                if (_myActiveInvite != value)
                {
                    _myActiveInvite = value;
                    OnPropertyChanged();
                }
            }
        }
    }
}
