﻿using System;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.NotificationContracts;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Events;
using Curse.FriendsService.Tester.UI.Messaging;
using Curse.FriendsService.Tester.UI.Notifications;
using Curse.FriendsService.Tester.Utilities;

namespace Curse.FriendsService.Tester.UI.ViewModels
{
    class NotificationsViewModel : BaseViewModel
    {
        private readonly TestClient _client;
        private readonly FriendsList _friendsList;
        private readonly GroupsList _groupsList;
        private readonly UserViewModel _currentUser;

        public DispatchedObservableCollection<BaseNotification> Notifications { get; private set; }

        public NotificationsViewModel(TestClient client, UserViewModel currentUser, FriendsList friendsList, GroupsList groupsList, IMessenger messenger)
        {
            _client = client;
            _currentUser = currentUser;
            _friendsList = friendsList;
            _groupsList = groupsList;

            Notifications = new DispatchedObservableCollection<BaseNotification>();

            client.Events.Register<MessageSent>(client_InstantMessageSent);
            client.Events.Register<FriendMessageReceived>(client_InstantMessageReceived);
            client.Events.Register<GroupMessageReceived>(client_GroupMessageReceived);
            client.Events.Register<FriendCallInvitationReceived>(client_VoiceInvitationReceived);
            client.Events.Register<GroupCallInvitationReceived>(client_GroupVoiceInvitationReceived);

            client.Events.Register<FriendCallDeclined>(client_FriendVoiceInviteDeclined);
            client.Events.Register<GroupCallDeclined>(client_GroupVoiceInviteDeclined);

            messenger.Register<NotificationMessage>(HandleNotificationMessage);
        }

        void client_GroupVoiceInviteDeclined(GroupCallDeclined e)
        {
            AddNotification(new SimpleMessageNotification
            {
                Timestamp = e.TimeStamp,
                Message = string.Format("{0} ({1}) declined a voice call to Group {2} ({3}) with invite code {4}.",
                GetUserName(e.SenderID), e.SenderID, GetGroupTitle(e.GroupID), e.GroupID, e.InviteUrl)
            });
        }

        void client_FriendVoiceInviteDeclined(FriendCallDeclined e)
        {
            AddNotification(new SimpleMessageNotification
            {
                Timestamp = DateTime.UtcNow,
                Message = string.Format("{0} ({1}) Declined a friend-to-friend call with invite code {2}.",
                    GetUserName(e.SenderID), e.SenderID, e.InviteUrl)
            });
        }

        void client_GroupVoiceInvitationReceived(GroupCallInvitationReceived e)
        {
            AddNotification(new CallNotification(_client, e.InviteUrl, null, e.RequestorID, _groupsList.Groups.First(g => g.GroupID == e.GroupID)));
        }

        void client_VoiceInvitationReceived(FriendCallInvitationReceived e)
        {
            AddNotification(new CallNotification(_client, e.InviteUrl, e.AccessToken, e.SenderID));
        }

        private void HandleNotificationMessage(NotificationMessage message)
        {
            AddNotification(new SimpleMessageNotification
            {
                Timestamp = DateTime.UtcNow,
                Message = message.Message
            });
        }

        void client_GroupMessageReceived(GroupMessageReceived e)
        {
            AddNotification(new SimpleMessageNotification
            {
                Timestamp = e.Timestamp,
                Message = string.Format("{0} ({1}) in {2} ({3}): {4}",GetUserName(e.SenderID),e.SenderID,GetGroupTitle(e.GroupID),e.GroupID,ResolveMentions(e.Message))
            });
        }

        private string ResolveMentions(string message)
        {
            return MentionsHelper.ResolveMentionsForDisplay(message, _friendsList.Friends.Where(f => f.Nickname != null).ToDictionary(f => f.UserID, f => f.Nickname));
        }

        private void client_InstantMessageReceived(FriendMessageReceived e)
        {
            AddNotification(new SimpleMessageNotification
            {
                Timestamp = e.Timestamp,
                Message = string.Format("{0} ({1}): {2}", GetUserName(e.SenderID), e.SenderID, e.Message)
            });
        }

        void client_InstantMessageSent(MessageSent e)
        {
            if (e.Status == DeliveryStatus.Successful)
            {
                return;
            }

            AddNotification(new SimpleMessageNotification
            {
                Timestamp = DateTime.UtcNow,
                Message = string.Format("Failed to send a message to {0} ({1}). {2}", GetUserName(e.RecipientID), e.RecipientID, e.Status)
            });
        }

        private string GetUserName(int id)
        {
            if (_currentUser.UserID == id)
            {
                return _currentUser.Username;
            }

            var friend = _friendsList.Friends.FirstOrDefault(f => f.UserID == id);
            if (friend != null)
            {
                return string.IsNullOrWhiteSpace(friend.Nickname)
                    ? friend.Username
                    : friend.Nickname;
            }

            string error;
            UserProfileResponse profile;
            return _client.Execute(new GetProfile(id), out profile, out error) ? profile.Username : "Unknown";
        }

        private string GetGroupTitle(Guid id)
        {
            var group = _groupsList.Groups.FirstOrDefault(g => g.GroupID == id);
            return group == null
                ? "Unknown"
                : group.Title;
        }

        private void AddNotification(BaseNotification notification)
        {
            var idx = Notifications.SkipWhile((item, index) => item.Timestamp < notification.Timestamp)
                .Select((item, index) => index).FirstOrDefault();
            Notifications.Insert(idx, notification);
        }

    }
}
