﻿using System.Collections.ObjectModel;
using System.Windows.Input;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Events;
using Curse.FriendsService.Tester.UI.Messaging;

namespace Curse.FriendsService.Tester.UI.ViewModels
{
    class StatusViewModel : BaseViewModel
    {
        private readonly TestClient _client;
        private readonly IMessenger _messenger;

        public StatusViewModel(TestClient client, IMessenger messenger, UserViewModel currentUser, FriendsList friendsList, GroupsList groupsList)
        {
            _client = client;
            _messenger = messenger;

            LogoutCommand = new RelayCommand(Logout);
            HangUpCommand = new RelayCommand(HangUp);

            CurrentUser = currentUser;
            Friends = friendsList.Friends;
            Groups = groupsList.Groups;

            Friends.CollectionChanged += (o,e) => OnPropertyChanged("Friends");

            client.Events.Register<UserChanged>(client_UserChanged);

            client.Events.Register<JoinedCall>(VoiceRuntimeClient_Connected);
            client.Events.Register<LeftCall>(VoiceRuntimeClient_Disconnected);
        }

        void client_UserChanged(UserChanged e)
        {
            if (e.User.UserID == _client.UserID)
            {
                CurrentUser.ConnectionStatus = e.User.ConnectionStatus;
                CurrentUser.StatusMessage = e.User.CustomStatusMessage;
                CurrentUser.GameID = e.User.CurrentGameID;
                CurrentUser.GameStatusMessage = string.IsNullOrWhiteSpace(e.User.CurrentGameStatusMessage) ? null : e.User.CurrentGameStatusMessage;
                CurrentUser.AvatarUrl = e.User.AvatarUrl;
            }
        }

        public ICommand LogoutCommand { get; private set; }

        public ICommand HangUpCommand { get; private set; }

        public void Logout()
        {
            string throwaway;
            _client.Logout(out throwaway);
            _messenger.Raise(new LogoutMessage());
        }

        public void HangUp()
        {
            string message;
            _client.Execute(new HangUp(), out message);
        }

        public UserViewModel CurrentUser { get; private set; }

        public ObservableCollection<FriendViewModel> Friends { get; private set;}
        public ObservableCollection<GroupViewModel> Groups { get; private set; }

        private string _currentCall;
        public string CurrentCall
        {
            get { return _currentCall; }
            set
            {
                if (_currentCall != value)
                {
                    _currentCall = value;
                    OnPropertyChanged();
                }
            }
        }

        void VoiceRuntimeClient_Connected(JoinedCall value)
        {
            CurrentCall = value.InviteCode;
        }

        void VoiceRuntimeClient_Disconnected(LeftCall value)
        {
            _messenger.Raise(new NotificationMessage
            {
                Message = string.Format("Call {0} ended. Initiated by {1}. Reason: {2}", CurrentCall, value.InitiatingUserID, value.Reason)
            });
            CurrentCall = null;
        }
    }
}
