﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Events;
using Curse.FriendsService.Tester.Groups;

namespace Curse.FriendsService.Tester
{
    public class GroupCache
    {
        private readonly TestClient _client;

        private readonly ConcurrentDictionary<Guid, Group> _children;

        //public GroupCache(GroupMemberNotification groupMembership, TestClient client):this(client)
        //{
        //    Members = new ConcurrentDictionary<int, GroupMemberNotification>();
        //    RootGroup = new Group(groupMembership);
        //    _children = new ConcurrentDictionary<Guid, Group>();
        //}

        public GroupCache(GroupNotification groupNotification, TestClient client)
            : this(client)
        {
            Members = new ConcurrentDictionary<int, GroupMemberNotification>();
            RootGroup = new Group(groupNotification);
            _children = new ConcurrentDictionary<Guid, Group>();
        }

        private GroupCache(TestClient client)
        {
            _client = client;

            client.Events.Register<GroupChanged>(GroupChanged);
        }

        private void GroupChanged(GroupChanged notification)
        {
            var notificationGroup = notification.Group;

            if (notificationGroup.RootGroupID != RootGroup.GroupID)
            {
                return;
            }

            switch (notification.ChangeType)
            {
                case  GroupChangeType.CreateGroup:
                    var group = AddOrUpdateGroup(notificationGroup);
                    _client.Events.Raise(new GroupAdded {Group = group});
                    break;

                case GroupChangeType.AddUsers:

                    foreach (var newUser in notification.Members)
                    {
                        AddOrUpdateMember(newUser);
                    }
                    Notify(MembersAdded, new GroupMemberEventArgs
                    {
                        AffectedGroup = this,
                        AffectedMembers = notification.Members
                    });

                    break;

                case GroupChangeType.RemoveUsers:

                    foreach (var removedUser in notification.Members)
                    {
                        RemoverMember(removedUser.UserID);
                    }
                    Notify(MembersRemoved, new GroupMemberEventArgs
                    {
                        AffectedGroup = this,
                        AffectedMembers = notification.Members
                    });

                    break;

                case GroupChangeType.UpdateUsers:

                    foreach (var u in notification.Members)
                    {
                        AddOrUpdateMember(u);
                    }
                    Notify(MembersUpdated, new GroupMemberEventArgs
                    {
                        AffectedGroup = this,
                        AffectedMembers = notification.Members
                    });

                    break;

                case GroupChangeType.ChangeInfo:
                {
                    AddOrUpdateGroup(notificationGroup);
                    Notify(InfoChanged, new GroupEventArgs {AffectedGroup = this});
                    break;
                }

                case GroupChangeType.GroupReorganized:
                {
                    AddOrUpdateGroup(notificationGroup);
                    break;
                }
                case GroupChangeType.VoiceSessionStarted:
                {
                    AddOrUpdateGroup(notificationGroup);
                    Notify(InfoChanged, new GroupEventArgs {AffectedGroup = this});
                    break;
                }
            }

        }

        public void AddChild(Group group)
        {
            _children.TryAdd(group.GroupID, group);
        }

        public void AddMember(GroupMemberNotification member)
        {
            Members.TryAdd(member.UserID, member);
        }

        public GroupMemberNotification UpdateMember(GroupMemberNotification update)
        {
            GroupMemberNotification found;

            if (Members.TryGetValue(update.UserID, out found))
            {
                found.Username = update.Username;
            }

            return found;
        }       

        public Group AddOrUpdateGroup(GroupNotification notification)
        {
            if (notification.GroupID == RootGroup.GroupID)
            {
                RootGroup.UpdateFromNotification(notification);
                return RootGroup;
            }

            var found = _children.AddOrUpdate(notification.GroupID, new Group(notification), (groupID, existing) =>
            {
                existing.UpdateFromNotification(notification);
                return existing;
            });

            return found;
        }

        public GroupMemberNotification AddOrUpdateMember(GroupMemberNotification update)
        {
            GroupMemberNotification found;
            if (Members.TryGetValue(update.UserID, out found))
            {
                found.Username = update.Username;
            }
            else
            {
                found = update;
                Members.TryAdd(update.UserID, found);
            }

            return found;
        }

        public void RemoveChildGroup(Guid groupID)
        {
            Group found;
            _children.TryRemove(groupID, out found);
        }

        public Group GetGroup(Guid groupID)
        {
            if (groupID == RootGroup.GroupID)
            {
                return RootGroup;
            }

            Group found;
            if (_children.TryGetValue(groupID, out found))
            {
                return found;
            }

            return null;
        }

        public bool ChildExists(GroupNotification notification)
        {
            return _children.ContainsKey(notification.GroupID);
        }

        public void AddMembers(GroupMemberNotification[] members)
        {
            foreach (var member in members)
            {
                AddMember(member);
            }
        }

        public GroupMemberNotification RemoverMember(int userID)
        {
            GroupMemberNotification found;
            Members.TryRemove(userID, out found);    
            return found;
        }

        public GroupMemberNotification GetMember(int userID)
        {
            GroupMemberNotification found;
            Members.TryGetValue(userID, out found);
            return found;
        }

        public bool HasLoadedDetails { get; set; }

        public Group RootGroup { get; set; }

        public ConcurrentDictionary<int, GroupMemberNotification> Members { get; set; }


        public GroupInvitation[] AllGroupInvites { get; set; }

        public Dictionary<GroupPermissions, int> Permissions { get; private set; } 

        public bool LoadDetails(out string errorMessage)
        {
            return LoadDetails(new GroupDetailsInfo
            {
                IncludeDeletedChannels = true,
                IncludePermissions = true,
                IncludeRoleNames = true
            }, out errorMessage);
        }

        public Group[] GetChildren(Func<IEnumerable<Group>, IEnumerable<Group>> filter, bool includeDeleted = false)
        {
            IEnumerable<Group> groups = _children.Values;
            if (!includeDeleted)
            {
                groups = groups.Where(g => g.Status != GroupStatus.Deleted);
            }

            return filter(groups).ToArray();
        }

        public Group[] GetChildren(bool includeDeleted = false)
        {
            return GetChildren(groups => groups, includeDeleted);
        }

        public bool LoadDetails(GroupDetailsInfo info, out string errorMessage)
        {
            GetGroupDetailsResponse response;
            if (!_client.Execute(new GetGroupDetails(RootGroup.GroupID)
            {
                IncludePermissions = info.IncludePermissions,
                IncludeRoleNames = info.IncludeRoleNames,
                IncludeDeletedChannels = info.IncludeDeletedChannels
            }, out response, out errorMessage))
            {
                return false;
            }
            else
            {
                AllGroupInvites = GetGroupInvites(false).ToArray();

                foreach (var member in response.Members)
                {
                    AddOrUpdateMember(member);
                }

                foreach (var groupNotification in response.Groups)
                {
                    AddOrUpdateGroup(groupNotification);
                }

                HasLoadedDetails = true;

                errorMessage = null;

                return true;
            }
        }

        public IEnumerable<GroupInvitation> GetGroupInvites(bool mineOnly)
        {
            string error;
            GroupInvitation[] invites;
            if (!_client.Execute(new GetGroupInvites(RootGroup.GroupID) {ActiveOnly = true, MineOnly = mineOnly}, out invites, out error))
            {
                invites = new GroupInvitation[0];
            }
            return invites;
        }

        public event EventHandler<GroupMemberEventArgs> MembersAdded;
        public event EventHandler<GroupMemberEventArgs> MembersRemoved;
        public event EventHandler<GroupMemberEventArgs> MembersUpdated;
        public event EventHandler<GroupEventArgs> InfoChanged;

        private void Notify<T>(EventHandler<T> handler, T args)
        {
            if (handler != null)
            {
                handler(this, args);
            }
        }
    }
}
