﻿using System;
using System.Collections.Concurrent;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Events;
using Curse.Logging;

namespace Curse.FriendsService.Tester.Groups
{
    public class GroupCollection
    {
        private readonly ConcurrentDictionary<Guid, GroupCache> _caches = new ConcurrentDictionary<Guid, GroupCache>();
        private readonly TestClient _client;
        
        internal GroupCollection(TestClient client)
        {
            _client = client;

            client.Events.Register<GroupChanged>(notificationClient_GroupChanged);
        }

        public void Reload()
        {
            string error;
            GetMyGroupsResponse response;
            if (_client.Execute(new GetMyGroups(), out response, out error))
            {
                foreach (var id in _caches.Keys.Except(response.Groups.Select(g => g.GroupID)))
                {
                    GroupCache temp;
                    _caches.TryRemove(id, out temp);
                }

                foreach (var groupMembership in response.Groups.Where(g => g.GroupID == g.RootGroupID))
                {
                    _caches.TryAdd(groupMembership.GroupID, new GroupCache(groupMembership, _client));
                }
            }
        }

        internal void Clear()
        {
            _caches.Clear();
        }

        public GroupCache[] Groups { get { return _caches.Values.ToArray(); } }

        public GroupCache this[Guid id]
        {
            get { return _caches[id]; }
        }

        public bool TryGetValue(Guid id, out GroupCache groupCache)
        {
            return _caches.TryGetValue(id, out groupCache);
        }

        private void notificationClient_GroupChanged(GroupChanged notification)
        {
            var notificationGroup = notification.Group;

            GroupCache cache;

            if (!_caches.TryGetValue(notificationGroup.RootGroupID, out cache) &&
                notificationGroup.GroupID == notificationGroup.RootGroupID && notification.Members.Any(m => m.UserID == _client.UserID))
            {
                cache = new GroupCache(notificationGroup, _client);
                cache.AddMembers(notification.Members);
                if (_caches.TryAdd(cache.RootGroup.GroupID, cache))
                {
                    NotifyGroupChanged(Added, new GroupEventArgs {AffectedGroup = cache});
                    _client.Events.Raise(new GroupAdded {Group = cache.RootGroup});
                }
            }

            if (cache == null)
            {
                Console.WriteLine("Received a notification for an unknown parented group!");
                Console.ReadKey(true);
                return;
            }

            switch (notification.ChangeType)
            {
                case GroupChangeType.CreateGroup:
                    if (!_caches.TryGetValue(notificationGroup.GroupID, out cache) && notificationGroup.GroupID == notificationGroup.RootGroupID)
                    {
                        cache = new GroupCache(notificationGroup, _client);
                        cache.AddMembers(notification.Members);
                        if (_caches.TryAdd(cache.RootGroup.GroupID, cache))
                        {
                            NotifyGroupChanged(Added, new GroupEventArgs {AffectedGroup = cache});
                            _client.Events.Raise(new GroupAdded {Group = cache.RootGroup});
                        }
                    }
                    break;
                case GroupChangeType.AddUsers:
                    if (!_caches.TryGetValue(notificationGroup.GroupID, out cache) &&
                        notificationGroup.GroupID == notificationGroup.RootGroupID && notification.Members.Any(m => m.UserID == _client.UserID))
                    {
                        cache = new GroupCache(notificationGroup, _client);
                        cache.AddMembers(notification.Members);
                        if (_caches.TryAdd(cache.RootGroup.GroupID, cache))
                        {
                            NotifyGroupChanged(Added, new GroupEventArgs {AffectedGroup = cache});
                            _client.Events.Raise(new GroupAdded {Group = cache.RootGroup});
                        }

                    }
                    break;
                case GroupChangeType.RemoveGroup:
                    if (_caches.TryGetValue(notificationGroup.GroupID, out cache)
                        && _caches.TryRemove(notificationGroup.GroupID, out cache))
                    {
                        NotifyGroupChanged(Removed, new GroupEventArgs {AffectedGroup = cache});
                    }
                    break;

                case GroupChangeType.RemoveUsers:

                    if (_caches.TryGetValue(notificationGroup.GroupID, out cache) &&
                        notification.Members.Any(p => p.UserID == _client.UserID) &&
                        _caches.TryRemove(cache.RootGroup.GroupID, out cache))
                    {
                        NotifyGroupChanged(Removed, new GroupEventArgs {AffectedGroup = cache});
                    }
                    break;
            }
        }

        public event EventHandler<GroupEventArgs> Added;
        public event EventHandler<GroupEventArgs> Removed;
        public event EventHandler<GroupMemberEventArgs> MembersRemoved;

        private void NotifyGroupChanged<T>(EventHandler<T> @event, T args)
        {
            if (@event != null)
            {
                @event(this, args);
            }
        }
    }
}
