﻿using System;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Events;

namespace Curse.FriendsService.Tester.People
{
    public class Friend : User
    {
        private readonly TestClient _client;

        private Friend(TestClient client) : base(client)
        {
            _client = client;

            client.Events.Register<FriendshipChanged>(client_FriendshipUpdated);
        }

        public Friend(TestClient client, FriendshipContract contract) : this(client)
        {
            UpdateFromNotification(contract);
        }

        public Friend(TestClient client, Friendship friend) : this(client)
        {
            UserID = friend.OtherUserID;
            Username = friend.OtherUsername;
            ConnectionStatus = friend.OtherUserConnectionStatus;
            Nickname = friend.OtherUserNickname;
            FriendshipStatus = friend.Status;
            Favorite = friend.IsFavorite;
            StatusMessage = friend.OtherUserStatusMessage;
            GameID = friend.OtherUserGameID;
            GameStatusMessage = friend.OtherUserGameStatusMessage;
        }

        public void ConfirmFriendship()
        {
            string error;
            _client.Execute(new ConfirmFriend(UserID), out error);
        }

        public void DeclineFriendship(bool block)
        {
            string error;
            _client.Execute(new DeclineFriendRequest(UserID) {BlockFutureRequests = block}, out error);
        }

        public string Nickname { get; set; }

        public FriendshipStatus FriendshipStatus { get; set; }

        public bool Favorite { get; set; }

        public MutualFriend[] MutualFriends { get; private set; }

        protected bool Equals(Friend other)
        {
            return UserID == other.UserID;
        }

        protected override void UpdateProfileFields(UserProfileResponse profile)
        {
            MutualFriends = profile.MutualFriends.Select(f => new MutualFriend(_client) { UserID = f.UserID, MutualFriendCount = f.MutualUserIDs.Length }).ToArray();
        }

        public override bool Equals(object obj)
        {
            if (ReferenceEquals(null, obj)) return false;
            if (ReferenceEquals(this, obj)) return true;
            var other = obj as Friend;
            return other != null && Equals(other);
        }

        public override int GetHashCode()
        {
            return UserID;
        }

        private void UpdateFromNotification(FriendshipContract contract)
        {
            UserID = contract.OtherUserID;
            Username = contract.OtherUsername;
            ConnectionStatus = contract.OtherUserConnectionStatus;
            Nickname = contract.OtherUserNickname;
            FriendshipStatus = contract.Status;
            Favorite = contract.IsFavorite;
            StatusMessage = contract.OtherUserStatusMessage;
            GameID = contract.OtherUserGameID;
            GameStatusMessage = contract.OtherUserGameStatusMessage;
        }

        void client_FriendshipUpdated(FriendshipChanged e)
        {
            if (UserID != e.Friendship.OtherUserID)
            {
                return;
            }
            // Update
            UpdateFromNotification(e.Friendship);
            Notify(Updated, new FriendEventArgs {AffectedFriend = this});
        }
        public event EventHandler<FriendEventArgs> Updated;

        private void Notify<T>(EventHandler<T> handler, T args)
        {
            if (handler != null)
            {
                handler(this, args);
            }
        }

    }
}
