﻿using System;
using System.Collections.Concurrent;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Events;

namespace Curse.FriendsService.Tester.People
{
    public class FriendCollection
    {
        private readonly ConcurrentDictionary<int, Friend> _friends = new ConcurrentDictionary<int, Friend>(); 
        private readonly TestClient _client;

        public Friend[] Friends
        {
            get { return _friends.Values.ToArray(); }
        }

        public FriendCollection(TestClient client)
        {
            _client = client;

            client.Events.Register<FriendshipChanged>(client_FriendshipChanged);
            client.Events.Register<FriendshipRemoved>(client_FriendshipRemoved);
        }

        public void Reload()
        {
            string error;
            GetMyFriendsResponse friends;
            if (_client.Execute(new GetMyFriends(), out friends, out error))
            {
                foreach (var id in _friends.Keys.Except(friends.Friends.Select(f=>f.OtherUserID)))
                {
                    Friend temp;
                    _friends.TryRemove(id, out temp);
                }
                foreach (var friend in friends.Friends)
                {
                    _friends.TryAdd(friend.OtherUserID, new Friend(_client, friend));
                }
            }
        }

        internal void Clear()
        {
            _friends.Clear();
        }

        public bool IsFriend(int otherUserID)
        {
            return _friends.ContainsKey(otherUserID);
        }

        public bool IsFriend(string userName)
        {
            return _friends.Any(p => p.Value.Username == userName);
        }

        public bool TryGetValue(int otherUserID, out Friend friend)
        {
            return _friends.TryGetValue(otherUserID, out friend);
        }

        void client_FriendshipRemoved(FriendshipRemoved e)
        {
            Friend removed;
            if (_friends.TryRemove(e.FriendID, out removed))
            {
                _client.Events.Raise(new FriendRemoved {Removed = removed});
                Notify(Removed, new FriendEventArgs {AffectedFriend = removed});
            }
        }

        void client_FriendshipChanged(FriendshipChanged e)
        {
            var friend = Friends.FirstOrDefault(f => f.UserID == e.Friendship.OtherUserID);
            if (friend == null)
            {
                friend = new Friend(_client, e.Friendship);
                // Add
                if (_friends.TryAdd(e.Friendship.OtherUserID, friend))
                {
                    _client.Events.Raise(new FriendAdded {Friend = friend});
                    Notify(Added, new FriendEventArgs {AffectedFriend = friend});
                }
            }
        }

        public event EventHandler<FriendEventArgs> Added;
        public event EventHandler<FriendEventArgs> Removed;

        private void Notify<T>(EventHandler<T> handler, T args)
        {
            if (handler != null)
            {
                handler(this, args);
            }
        }
    }
}
