﻿using System;
using System.Net;
using System.Reflection;
using System.Security.Cryptography;
using System.Text;
using System.Threading;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.FriendsService.Tester.Actions;
using Curse.FriendsService.Tester.Events;
using Curse.FriendsService.Tester.Groups;
using Curse.FriendsService.Tester.People;
using Curse.ServiceAuthentication.Models;
using Curse.SocketInterface;

namespace Curse.FriendsService.Tester
{
    public class TestClient
    {
        private static int _clientNumber;
        private readonly Guid _machineKey;

        internal TestLoginServiceClient LoginClient { get; private set; }

        internal TestVoiceServiceClient VoiceServiceClient { get; private set; }

        internal TestFriendsWebServiceClient WebClient { get; private set; }

        internal TestFriendsNotificationClient NotificationClient { get; private set; }

        internal TestVoiceRuntimeClient VoiceRuntimeClient { get; private set; }

        public TestClient(ServiceConnectionInfo connectionInfo, bool generatedGuid = false)
        {
            _machineKey = generatedGuid ? GenerateMachineKey() : new Guid("88888888-8888-8888-8888-888888888888");
            Events = new EventDispatcher();

            WebClient = new TestFriendsWebServiceClient(connectionInfo.FriendsInsecure, connectionInfo.FriendsSecure);
            WebClient.EnableCallLogging = false;

            NotificationClient = new TestFriendsNotificationClient(connectionInfo.UseLocalServices, Events);
            VoiceServiceClient = new TestVoiceServiceClient(connectionInfo.VoiceSecure, connectionInfo.VoiceInsecure);
            VoiceServiceClient.EnableCallLogging = false;

            VoiceRuntimeClient = new TestVoiceRuntimeClient(VoiceServiceClient, Events);
            LoginClient = new TestLoginServiceClient(connectionInfo.LoginSecure, connectionInfo.LoginInsecure);

            GroupCache = new GroupCollection(this);
            FriendCache = new FriendCollection(this);
        }

        private static Guid GenerateMachineKey()
        {
            var clientNumber = Interlocked.Increment(ref _clientNumber);

            // This is primarily to ensure unique machine keys per client in the same application, but reduce the overall number of clientendpoints created
            // It will not be unique across multiple instances of the same process (e.g., 2 console apps open)
            var identifier = Assembly.GetEntryAssembly().GetName().Name + clientNumber;
            var hash = MD5.Create().ComputeHash(Encoding.UTF8.GetBytes(identifier));

            // UUID for MD5 hash has the format xxxxxxxx-xxxx-3xxx-yxxx-xxxxxxxxxx where 3 identifies MD5 and y is a 10xx binary value (8, 9, A, or B hex)
            hash[7] &= 0x3F;
            hash[7] |= 0x30;

            hash[8] |= 0x80;
            hash[8] &= 0xBF;

            return new Guid(hash);
        }

        public EventDispatcher Events { get; private set; }

        public GroupCollection GroupCache { get; private set; }
        
        public FriendCollection FriendCache { get; private set; }


        public CurrentUser CurrentUser { get; private set; }

        public int UserID { get { return CurrentUser==null?0:CurrentUser.UserID; } }

        public string Username { get { return CurrentUser==null?null:CurrentUser.Username; } }
        
        public bool IsLoggedIn { get { return CurrentUser != null; } }

        public string CurrentCall { get { return VoiceRuntimeClient.CurrentInviteCode; } }

        public Guid MachineKey { get { return _machineKey; } }

        public bool Login(LoginInfo info, out string errorMessage)
        {
            if (IsLoggedIn)
            {
                errorMessage = "Already logged in. Log out first.";
                return false;
            }
            try
            {
                var currentUser = Authenticate(info.Credentials);

                var result = RegisterWithFriendsService(this, MachineKey, info.Device);

                NotificationClient.Connect(result.NotificationHostList, result.NotificationHostPorts, result.SessionID, currentUser.UserID, MachineKey, info.DesiredStatus);

                currentUser.ConnectionStatus = info.DesiredStatus;
                
                CurrentUser = currentUser;
                CurrentUser.RefreshProfile();
                GroupCache.Reload();
                FriendCache.Reload();

                Events.Raise(new LoggedIn
                {
                    User = CurrentUser
                });

                errorMessage = null;
                return true;
            }
            catch (Exception ex)
            {
                var sb = new StringBuilder();
                var e = ex;
                while (e != null)
                {
                    sb.Append(e.Message).Append(". ");
                    e = e.InnerException;
                }

                CurrentUser = null;
                errorMessage = sb.ToString();
                return false;
            }
        }

        private CurrentUser Authenticate(NetworkCredential credentials)
        {
            var resp = LoginClient.Call("Login", svc => svc.Login(new LoginRequest
            {
                Username = credentials.UserName,
                EncryptedPassword = ClientLogin.EncryptLocalString(credentials.Password)
            }));

            if (resp.Status != AuthenticationStatus.Success)
            {
                throw new InvalidOperationException("Login failed with status " + resp.Status);
            }

            var token = new TestAuthenticationToken(resp.Session.UserID, resp.Session.Token, null);
            WebClient.UpdateToken(token);
            VoiceServiceClient.UpdateToken(token);

            return new CurrentUser(this)
            {
                UserID = resp.Session.UserID,
                Username = resp.Session.Username,
                Token = resp.Session.Token
            };
        }

        private static RegisterSelfResponse RegisterWithFriendsService(TestClient client, Guid machineKey, Device device)
        {
            var resp = client.WebClient.Call("RegisterSelf", svc => svc.RegisterSelf(
                new RegisterSelfRequest
                {
                    MachineKey = machineKey,
                    Status = UserConnectionStatus.Online,
                    Platform = device.Platform,
                    DeviceID = device.DeviceID
                }));

            if (resp.Status != RegisterSelfStatus.Successful)
            {
                throw new Exception("Failed to register self with Friends Service.");
            }

            return resp;
        }

        public bool Logout(out string errorMessage)
        {
            if (!IsLoggedIn)
            {
                errorMessage = "Not currently logged in.";
                return false;
            }

            var userID = CurrentUser.UserID;
            NotificationClient.Disconnect(SocketDisconnectReason.UserInitiated);
            WebClient.UpdateToken(null);
            VoiceServiceClient.UpdateToken(null);

            CurrentUser = null;
            GroupCache.Clear();
            FriendCache.Clear();

            Events.Raise(new LoggedOut
            {
                UserID = userID
            });

            errorMessage = null;
            return true;
        }

        public bool Execute(BaseAction action, out string message)
        {
            return action.Execute(this, out message);
        }

        public bool Execute<T>(BaseAction<T> action, out T result, out string errorMessage)
        {
            return action.Execute(this, out result, out errorMessage);
        }
    }
}
