﻿using System;
using System.Collections.Generic;
using System.Text.RegularExpressions;

namespace Curse.FriendsService.Tester.Utilities
{
    public static class MentionsHelper
    {
        /// <summary>
        /// Resolves Mentions to their mapped userID and username
        /// </summary>
        /// <param name="message">The raw message text from the user</param>
        /// <param name="termMapping">A mapping from a search term (e.g., Nickname or Username) to a UserID and Username</param>
        /// <returns>Resolved message for sending to the service</returns>
        public static string ResolveMentionsForSend(string message, Dictionary<string,Tuple<int,string>> termMapping)
        {
            return Regex.Replace(message, string.Format("(?<initial>\\s@|^@)(?<term>{0})(?:\\b|$)", string.Join("|", termMapping.Keys)), match =>
            {
                var mappedItem = termMapping[match.Groups["term"].Value];
                return match.Result(string.Format("${{initial}}{0}:{1}", mappedItem.Item1, mappedItem.Item2));
            });
        }

        /// <summary>
        /// Resolves Mentions to their mapped display names.
        /// </summary>
        /// <param name="message">The raw message text from the server.</param>
        /// <param name="displayNameOverrides">Mappings from user ID to display name (e.g., Nickname)</param>
        /// <returns></returns>
        public static string ResolveMentionsForDisplay(string message, Dictionary<int, string> displayNameOverrides)
        {
            return Regex.Replace(message, "(?<mentionStart>\\s@|^@)(?<userID>\\d+):(?<username>\\w+)(?:\\b|$)", match =>
            {
                int userID;
                if (!int.TryParse(match.Groups["userID"].Value, out userID))
                {
                    return match.Value;
                }

                string displayName;
                if (!displayNameOverrides.TryGetValue(userID, out displayName))
                {
                    displayName = match.Groups["username"].Value;
                }

                return match.Result(string.Format("${{mentionStart}}({0})", displayName));
            });
        } 
    }
}
