﻿using System;
using System.Collections.Generic;
using Curse.Friends.Client;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.FriendsService.Tester.Events;
using Curse.SocketInterface;
using Curse.SocketMessages;

namespace Curse.FriendsService.Tester
{
    public class TestFriendsNotificationClient
    {
        private FriendsNotificationClient _notificationClient;
        private readonly bool _useLocalService;
        private readonly EventDispatcher _eventDispatcher;

        public TestFriendsNotificationClient(bool useLocalService, EventDispatcher eventDispatcher)
        {
            _eventDispatcher = eventDispatcher;
            _useLocalService = useLocalService;

            FriendsNotificationClient.InstantMessageNotification += _notificationClient_InstantMessageNotification;
            FriendsNotificationClient.InstantMessageResponse += _notificationClient_InstantMessageResponse;
            FriendsNotificationClient.FriendshipChangeNotification += _notificationClient_FriendshipChangeNotification;
            FriendsNotificationClient.FriendshipRemovedNotification += _notificationClient_FriendshipRemovedNotification;
            FriendsNotificationClient.UserChangeNotification += Instance_UserChangeNotification;
            FriendsNotificationClient.VoiceInvitationNotification += FriendsNotificationClient_VoiceInvitationNotification;
            FriendsNotificationClient.VoiceDeclineNotification += FriendsNotificationClient_VoiceDeclineNotification;
            FriendsNotificationClient.GroupMessageNotification += FriendsNotificationClient_GroupMessageNotification;
            FriendsNotificationClient.GroupChangeNotification += FriendsNotificationClientOnGroupChangeNotification;
            FriendsNotificationClient.OfflineMessageNotification += FriendsNotificationClientOnOfflineMessageNotification;
            FriendsNotificationClient.GroupVoiceDeclineNotification += FriendsNotificationClient_GroupVoiceDeclineNotification;
            FriendsNotificationClient.GroupVoiceInvitationNotification += _notificationClient_GroupVoiceInvitationNotification;
            FriendsNotificationClient.VoiceAcceptNotification += _notificationClient_VoiceAcceptNotification;
        }

        public void Connect(string[] hostList, int[] ports, string sessionID, int userID, Guid machineKey, UserConnectionStatus desiredStatus)
        {
            var connectionInfo = new NotificationConnectionInfo
            {
                Ports = ports,
                UserID = userID,
                MachineKey = machineKey.ToString(),
                NotificationHosts = _useLocalService ? new[] {"127.0.0.1"} : hostList,
                SessionID = sessionID,
                ClientVersion = "6.0.5000.0",
                DesiredStatus = desiredStatus
            };
            new Random((int) DateTime.UtcNow.Ticks).Shuffle(connectionInfo.NotificationHosts);

            JoinResponse resp;
            _notificationClient = FriendsNotificationClient.Connect(connectionInfo, out resp);

            if (resp.Status != JoinStatus.Successful)
            {
                throw new Exception("Connection to notification failed! Status=" + resp.Status);
            }

            _notificationClient.Disconnected += NotificationClient_Disconnected;
        }

        public void Disconnect(SocketDisconnectReason reason)
        {
            if (_notificationClient != null)
            {
                _notificationClient.Disconnect(reason);
            }
        }

        public void SendGroupMessage(Guid groupID, string message)
        {
            _notificationClient.SendGroupMessage(groupID, message, Guid.NewGuid());
        }

        public void SendVoiceDecline(int userID, string callCode)
        {
            _notificationClient.SendVoiceDecline(userID, callCode);
        }

        public void SendGroupVoiceDecline(Guid groupID, string callCode)
        {
            _notificationClient.SendGroupVoiceDecline(groupID, callCode);
        }

        public void SendInstantMessage(int userID, string message)
        {
            _notificationClient.SendInstantMessage(userID, message, Guid.NewGuid());
        }

        public void SendOfflineMessageRequest(DateTime since, int? friendID=null, Guid? groupID=null)
        {
            if (friendID.HasValue)
            {
                _notificationClient.SendOfflineMessageRequest(Guid.NewGuid(), since, friendID.Value);
            }
            else if (groupID.HasValue)
            {
                _notificationClient.SendOfflineMessageRequest(Guid.NewGuid(), since, groupID.Value);
            }
            else
            {
                _notificationClient.SendOfflineMessageRequest(Guid.NewGuid(), since);
            }
        }

        public void SendConversationRead(Guid groupID)
        {
            _notificationClient.SendConversationRead(groupID, DateTime.UtcNow);
        }

        public void SendConversationRead(int friendID)
        {
            _notificationClient.SendConversationRead(friendID, DateTime.UtcNow);
        }

        #region Notification Events

        void FriendsNotificationClient_VoiceDeclineNotification(object sender, EventArgs<VoiceDeclineNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<FriendCallDeclined>());
            }
        }

        void FriendsNotificationClient_GroupVoiceDeclineNotification(object sender, EventArgs<GroupVoiceDeclineNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<GroupCallDeclined>());
            }
        }

        void FriendsNotificationClient_VoiceInvitationNotification(object sender, EventArgs<VoiceInvitationNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<FriendCallInvitationReceived>());
            }
        }

        void _notificationClient_GroupVoiceInvitationNotification(object sender, EventArgs<GroupVoiceInvitationNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<GroupCallInvitationReceived>());
            }
        }

        void NotificationClient_Disconnected(object sender, SocketDisconnectEventArgs e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.ToDomainEvent<NotificationsDisconnected>());
            }
        }

        void Instance_UserChangeNotification(object sender, EventArgs<UserChangeNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<UserChanged>());
            }
        }

        void _notificationClient_FriendshipRemovedNotification(object sender, EventArgs<FriendshipRemovedNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<FriendshipRemoved>());
            }
        }

        void _notificationClient_FriendshipChangeNotification(object sender, EventArgs<FriendshipChangeNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<FriendshipChanged>());
            }
        }

        void _notificationClient_InstantMessageResponse(object sender, EventArgs<InstantMessageResponse> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<MessageSent>());
            }
        }

        void _notificationClient_InstantMessageNotification(object sender, EventArgs<InstantMessageNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<FriendMessageReceived>());
            }
        }

        private void FriendsNotificationClientOnOfflineMessageNotification(object sender, EventArgs<OfflineMessageNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<OfflineMessagesSynced>());
            }
        }

        private void FriendsNotificationClientOnGroupChangeNotification(object sender, EventArgs<GroupChangeNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<GroupChanged>());
            }
        }

        void FriendsNotificationClient_GroupMessageNotification(object sender, EventArgs<GroupMessageNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<GroupMessageReceived>());
            }
        }

        void _notificationClient_VoiceAcceptNotification(object sender, EventArgs<VoiceAcceptNotification> e)
        {
            if (sender == _notificationClient)
            {
                _eventDispatcher.Raise(e.Value.ToDomainEvent<CallAccepted>());
            }
        }

        #endregion    
    }
}
