﻿using System;
using System.Net;
using System.Threading.Tasks;
using Curse.FriendsService.Tester.CurseVoiceService;
using Curse.FriendsService.Tester.Events;
using Curse.FriendsService.Tester.People;
using Curse.SocketInterface;
using Curse.SocketMessages;
using Curse.Voice.Client;
using Curse.Voice.Contracts;
using Curse.Voice.Enums;

namespace Curse.FriendsService.Tester
{
    public class TestVoiceRuntimeClient
    {
        public static readonly Version ClientVersion = new Version(6, 2, 0, 0);

        private readonly TestVoiceServiceClient _serviceClient;
        private readonly EventDispatcher _eventDispatcher;

        private VoiceClient _client;

        public TestVoiceRuntimeClient(TestVoiceServiceClient serviceClient, EventDispatcher eventDispatcher)
        {
            _serviceClient = serviceClient;
            _eventDispatcher = eventDispatcher;

            _eventDispatcher.Register<LoggedOut>(HandleLoggedOut);
        }

        public bool IsInCall
        {
            get { return _client != null; }
        }

        public string CurrentInviteCode { get; private set; }

        public void JoinFriendCall(CurrentUser currentUser, string inviteUrl, int friendID, long accessToken)
        {
            if (IsInCall)
            {
                throw new InvalidOperationException("Already in a call, hang up before making another.");
            }

            Connect(currentUser, inviteUrl, friendID, null, VoiceInstanceType.Friend, accessToken);
        }

        public void JoinGroupCall(CurrentUser currentUser, string inviteUrl, Guid groupID, long accessToken)
        {
            if (IsInCall)
            {
                throw new InvalidOperationException("Already in a call, hang up before making another.");
            }

            Connect(currentUser, inviteUrl, null, groupID, VoiceInstanceType.Group, accessToken);
        }

        public void HangUp()
        {
            if (!IsInCall)
            {
                return;
            }

            _client.LeaveSession();

            Task.Delay(500).ContinueWith(_ =>
            {
                var client = _client;
                if (client != null && client.IsConnected)
                {
                    client.Disconnect(SocketDisconnectReason.UserInitiated);
                }
            });
        }

        private void Connect(CurrentUser currentUser, string inviteUrl, int? friendID, Guid? groupID, VoiceInstanceType callType, long? accessToken)
        {
            var resp = _serviceClient.Call("GetSession", s => s.GetVoiceSession(inviteUrl, ClientVersion));

            var connectionInfo = new VoiceConnectionInfo(new CodecInfo {Name = "Opus", PacketMilliseconds = 40})
            {
                GroupID = groupID ?? Guid.Empty,
                UserID = currentUser.UserID,
                FriendID = friendID??0,
                AvatarUrl = currentUser.AvatarUrl,
                GameID = resp.GameID,
                Port = resp.Port,
                AccessToken = accessToken,
                AuthToken = currentUser.Token,
                DisplayName = currentUser.Username,
                HostID = resp.HostID,
                HostName = resp.HostName,
                IPAddress = IPAddress.Parse(resp.IPAddress),
                InstanceID = resp.InstanceCode,
                InviteUrl = inviteUrl,
                OriginalDisplayName = currentUser.Username,
                RegionName = resp.RegionName,
                SessionType = callType,
            };


            if (resp.Status != GetVoiceSessionStatus.Successful)
            {
                throw new InvalidOperationException();
            }

            JoinSessionResponse joinResponse;
            _client = VoiceClient.Connect(new[] {resp.Port, 3784, 6100, 9987, 10011}, connectionInfo, false, ClientVersion.ToString(), out joinResponse);

            if (_client == null)
            {
                throw new InvalidOperationException();
            }

            if (_client.IsConnected)
            {
                _client.Disconnected += VoiceClient_Disconnected;
                _client.DisconnectedSession += VoiceClient_Disconnected;
                _client.UserDisconnected += VoiceClient_UserDisconnected;
                CurrentInviteCode = inviteUrl;
                _eventDispatcher.Raise(new JoinedCall {InviteCode = inviteUrl});
            }
            else
            {
                EndSession(_client, new UserDisconnectNotification());
            }
        }


        private void VoiceClient_UserDisconnected(object sender, EventArgs<UserDisconnectNotification> e)
        {
            if (e.Value.AffectedUserID == _client.UserID)
            {
                EndSession((VoiceClient) sender, e.Value);
            }
        }

        void VoiceClient_Disconnected(object sender, SocketDisconnectEventArgs e)
        {
            EndSession((VoiceClient) sender, new UserDisconnectNotification
            {
                Reason = UserDisconnectReason.CallEnded,
                Timestamp = DateTime.UtcNow
            });
        }

        private void EndSession(VoiceClient client, UserDisconnectNotification e)
        {
            client.Disconnected -= VoiceClient_Disconnected;
            client.DisconnectedSession -= VoiceClient_Disconnected;
            client.UserDisconnected -= VoiceClient_UserDisconnected;

            if (_client == client)
            {
                CurrentInviteCode = null;
                _client = null;
                _eventDispatcher.Raise(new LeftCall {InitiatingUserID = e.InitiatingUserID, Reason = e.Reason});
            }
        }

        private void HandleLoggedOut(LoggedOut value)
        {
            var client = _client;
            if (client != null)
            {
                client.Disconnect(SocketDisconnectReason.UserInitiated);
                EndSession(client, new UserDisconnectNotification {Reason = UserDisconnectReason.CallEnded});
            }
        }
    }
}
