﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Globalization;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.Friends.Enums;
using Newtonsoft.Json;

namespace Curse.PromoTool
{
    class Program
    {
        static void Main(string[] args)
        {
            if (Debugger.IsAttached)
            {
                Console.Write("Do you want to overwrite the args? (y|n)");
                if (Console.ReadKey(false).Key == ConsoleKey.Y)
                {
                    args = new[]
                    {
                        "21d41365-9e22-41f7-b710-f191a76b59e6",
                        "TSM TSM TSM ",
                        DateTime.Now.AddHours(-12).ToString(),
                        DateTime.Now.ToString(),
                        @"C:\temp\users.txt",
                    };
                }
                Console.WriteLine();
            }

            Guid promoID;
            string promoText;
            DateTime startTime;
            DateTime endTime;
            string fileName;

            if (!ParseArguments(args,out promoID, out promoText, out startTime, out endTime, out fileName))
            {
                Console.WriteLine("Usage: promoID promoText startTime endTime fileName");
                Console.WriteLine("promoID: The Guid for the promo");
                Console.WriteLine("promoText: The text that users need to say to be entered to win");
                Console.WriteLine("startTime: The time (formatted as MM/DD/YYYY hh:mm:ss AM|PM in local time) before which messages are ignored");
                Console.WriteLine("endTime: The time (formatted as MM/DD/YYYY hh:mm:ss AM|PM in local time) after which messages are ignored");
                Console.WriteLine("fileName: Path to the file to output users. Can be relative or absolute. Must not already exist.");
                return;
            }

            StorageConfiguration.Initialize("PromoTool");

            var promo = GroupPromo.GetLocal(promoID);
            if (promo == null)
            {
                Console.WriteLine("Promotion {0} does not exist.", promoID);
                return;
            }

            var users = new Dictionary<int,string>();
            foreach (var groupID in promo.GroupIDs)
            {
                var conversation = Conversation.GetByTypeAndID(ConversationType.Group, groupID);
                if (conversation == null)
                {
                    Console.WriteLine("Conversation for Group {0} not found. Skipping.", groupID);
                    continue;
                }

                var messages = conversation.History.Range(startTime.ToEpochMilliseconds(), endTime.ToEpochMilliseconds());
                foreach (var message in messages)
                {
                    if (message.Body.ToLowerInvariant().Contains(promoText.ToLowerInvariant().Trim()) &&
                        !users.ContainsKey(message.SenderID))
                    {
                        users.Add(message.SenderID, message.SenderName);
                    }
                }
            }

            try
            {
                using (var stream = new StreamWriter(File.Open(fileName,FileMode.Create,FileAccess.Write)))
                {
                    WriteUsersToFile(stream, users);
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("Error occurred while writing to {0}.", fileName);

                try
                {
                    var tempFileName = Path.GetTempFileName();
                    using (var stream = new StreamWriter(tempFileName))
                    {
                        WriteUsersToFile(stream, users);
                    }
                    Console.WriteLine("Wrote output to {0} instead.", tempFileName);
                }
                catch (Exception e)
                {
                    Console.WriteLine("Failed to dump to a temp file.");
                }
            }
        }

        private static void WriteUsersToFile(StreamWriter writer, Dictionary<int, string> users)
        {
            foreach (var user in users)
            {
                writer.WriteLine("{0} {1}", user.Value, user.Key);
            }
        }

        private static bool ParseArguments(string[] args, out Guid promoID, out string promoText, out DateTime startTime, out DateTime endTime, out string fileName)
        {
            promoID = Guid.Empty;
            promoText = null;
            startTime = DateTime.MinValue;
            endTime = DateTime.MinValue;
            fileName = null;

            if (args == null || args.Length < 5)
            {
                return false;
            }

            if (!Guid.TryParse(args[0], out promoID))
            {
                Console.WriteLine("Invalid Promo ID.");
                return false;
            }

            promoText = args[1];
            if (string.IsNullOrWhiteSpace(promoText))
            {
                Console.WriteLine("Promo Text cannot be empty.");
                return false;
            }

            if (!DateTime.TryParse(args[2], out startTime))
            {
                Console.WriteLine("Start date is invalid.");
                return false;
            }
            startTime = startTime.ToUniversalTime();

            if (!DateTime.TryParse(args[3], out endTime))
            {
                Console.WriteLine("End date is invalid.");
                return false;
            }
            endTime = endTime.ToUniversalTime();

            fileName = args[4];
            if (string.IsNullOrWhiteSpace(fileName))
            {
                Console.WriteLine("Filename is invalid.");
                return false;
            }

            return true;
        }
    }
}
