﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Aerospike.Client;
using Curse.CloudSearch;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Nest;

namespace Curse.Friends.ElasticIndexer
{
    class ExternalCommunityIndexer
    {
        public static void Index(ConfigurationRegion region)
        {
            Console.Clear();

            var aerospikeConfig = ExternalCommunity.GetConfiguration(region.ID);
            var elasticConfigs = SearchConfiguration.Configurations.Where(c => c.RegionIdentifier == region.ID).ToArray();

            var client = new ElasticClient(ExternalCommunitySearchManager.GetDefaultConnectionSettings(elasticConfigs));


            Console.WriteLine("Do you want to delete the index, before re-indexing? [Y|N]");

            var sw = new Stopwatch();
            if (Console.ReadKey(true).Key == ConsoleKey.Y)
            {
                // Replace template and delete index
                Console.Write("Replacing External Community template and deleting index...");
                sw.Start();
                var manager = new ExternalCommunitySearchManager();
                manager.DeleteTemplate();
                manager.SetupTemplate();
                client.DeleteIndex(ExternalCommunitySearchManager.IndexName);
                sw.Stop();
                Console.WriteLine("Done in {0}!", sw.Elapsed);
            }

            // Reindex
            Console.WriteLine("Indexing external communities...");
            Exception caught = null;
            sw.Restart();
            var count = 0;
            try
            {
                ExternalCommunity.BatchOperate(aerospikeConfig, 1000, communities =>
                {
                    try
                    {
                        var communityModels = new List<ExternalCommunitySearchModel>();
                        foreach (var community in communities)
                        {
                            count++;
                            communityModels.Add(new ExternalCommunitySearchModel(community));
                        }

                        BulkIndex(client, communityModels);

                        Console.Write("\rIndexed {0} communities.", count);
                    }
                    catch (Exception ex)
                    {
                        throw new AerospikeException.ScanTerminated(ex);
                    }
                });
            }
            catch (Exception ex)
            {
                caught = ex;
            }
            sw.Stop();

            Console.WriteLine();
            Console.WriteLine("Finished in {0}", sw.Elapsed);

            if (caught != null)
            {
                Console.WriteLine("Error occurred while indexing external communities! Press Enter to see the exception info");
                Console.ReadLine();

                Console.WriteLine("Outer exception:");
                Console.WriteLine(caught.Message);
                Console.WriteLine(caught.StackTrace);

                if (caught.InnerException != null)
                {
                    Console.WriteLine();
                    Console.WriteLine("Inner Exception:");
                    Console.WriteLine(caught.InnerException.Message);
                    Console.WriteLine(caught.InnerException.StackTrace);
                }
            }

            Console.WriteLine("Press Enter to continue");
            Console.ReadLine();
        }

        private static void BulkIndex(ElasticClient client, IEnumerable<ExternalCommunitySearchModel> communities)
        {
            var bulkRequest = new BulkRequest
            {
                Index = ExternalCommunitySearchManager.IndexName,
                Operations = communities.Select(m => (IBulkOperation)new BulkIndexOperation<ExternalCommunitySearchModel>(m)).ToArray()
            };
            client.Bulk(bulkRequest);
        }
    }
}
