﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Aerospike.Client;
using Curse.CloudSearch;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Nest;

namespace Curse.Friends.ElasticIndexer
{
    class GroupBannedUserIndexer
    {
        public static void Index(ConfigurationRegion region)
        {
            Console.Clear();

            var aerospikeConfig = GroupBannedUser.GetConfiguration(region.ID);
            var elasticConfigs = SearchConfiguration.Configurations.Where(c => c.RegionIdentifier == region.ID).ToArray();

            var client = new ElasticClient(GroupBannedUserSearchManager.GetDefaultConnectionSettings(elasticConfigs));


            Console.WriteLine("Do you want to delete the index, before re-indexing? [Y|N]");

            var sw = new Stopwatch();
            if (Console.ReadKey(true).Key == ConsoleKey.Y)
            {
                // Replace template and delete index
                Console.Write("Replacing Group Banned User template and deleting index...");
                sw.Start();
                var manager = new GroupBannedUserSearchManager();
                manager.DeleteTemplate();
                manager.SetupTemplate();
                client.DeleteIndex(GroupBannedUserSearchManager.IndexName);
                sw.Stop();
                Console.WriteLine("Done in {0}!", sw.Elapsed);
            }

            // Reindex
            Console.WriteLine("Indexing bans...");
            Exception caught = null;
            sw.Restart();
            var count = 0;
            var serverCount = 0;
            try
            {
                GroupBannedUser.BatchOperate(aerospikeConfig, 1000, bans =>
                {
                    try
                    {
                        var banModels = new List<GroupBannedUserSearchModel>();
                        foreach (var ban in bans)
                        {
                            count++;
                            serverCount++;
                            banModels.Add(new GroupBannedUserSearchModel(ban));
                        }

                        BulkIndex(client, banModels);

                        Console.Write("\rIndexed {0} bans.", count);
                    }
                    catch (Exception ex)
                    {
                        throw new AerospikeException.ScanTerminated(ex);
                    }
                });
            }
            catch (Exception ex)
            {
                caught = ex;
            }
            sw.Stop();

            Console.WriteLine();
            Console.WriteLine("Finished in {0}", sw.Elapsed);

            if (caught != null)
            {
                Console.WriteLine("Error occurred while indexing bans! Press Enter to see the exception info");
                Console.ReadLine();

                Console.WriteLine("Outer exception:");
                Console.WriteLine(caught.Message);
                Console.WriteLine(caught.StackTrace);

                if (caught.InnerException != null)
                {
                    Console.WriteLine();
                    Console.WriteLine("Inner Exception:");
                    Console.WriteLine(caught.InnerException.Message);
                    Console.WriteLine(caught.InnerException.StackTrace);
                }
            }

            Console.WriteLine("Press Enter to continue");
            Console.ReadLine();
        }

        private static void BulkIndex(ElasticClient client, IEnumerable<GroupBannedUserSearchModel> bans)
        {
            var bulkRequest = new BulkRequest
            {
                Index = GroupBannedUserSearchManager.IndexName,
                Operations = bans.Select(m => (IBulkOperation)new BulkIndexOperation<GroupBannedUserSearchModel>(m)
                {
                    Routing = m.GroupID.ToString(),
                }).ToArray()
            };
            client.Bulk(bulkRequest);
        }
    }
}
