﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Nest;

namespace Curse.Friends.ElasticIndexer
{
    class GroupIndexer
    {
        public static void Index(ConfigurationRegion region)
        {
            Console.Clear();

            var aerospikeConfig = Group.GetConfiguration(region.ID);
            var elasticConfigs = SearchConfiguration.Configurations.Where(c => c.RegionIdentifier == region.ID).ToArray();
            var sw = new Stopwatch();

            // Replace template and delete index
            Console.Write("Replacing Group template and deleting index...");
            sw.Start();
            var client = new ElasticClient(GroupSearchManager.GetDefaultConnectionSettings(elasticConfigs));
            var manager = new GroupSearchManager();
            manager.DeleteTemplate();
            manager.SetupTemplate();
            client.DeleteIndex(GroupSearchManager.IndexName);
            sw.Stop();
            Console.WriteLine("Done in {0}!", sw.Elapsed);

            // Reindex
            Console.WriteLine("Indexing groups...");
            Exception caught = null;
            sw.Restart();
            var indexed = 0;
            var count = 0;
            try
            {
                var defaultSettings = new GroupSearchSettings
                {
                    IsSearchable = true,
                    Games = new HashSet<int>(),
                    SearchTags = new HashSet<int>(),
                    Description = string.Empty,
                    MatchAllGames = false
                };
                Group.BatchOperate(aerospikeConfig, 1000, groups =>
                {
                    try
                    {
                        var allGroups = groups.ToArray();
                        var allSettings = GroupSearchSettings.MultiGetLocal(allGroups.Select(g => new KeyInfo(g.GroupID))).ToDictionary(g => g.GroupID);
                        var allAvatars = Avatar.MultiGetLocal(allGroups.Select(g => new KeyInfo((int) AvatarType.Group, g.GroupID.ToString()))).ToDictionary(a => new Guid(a.EntityID));
                        var allCovers = Avatar.MultiGetLocal(allGroups.Select(g => new KeyInfo((int) AvatarType.GroupCover, g.GroupID.ToString()))).ToDictionary(a => new Guid(a.EntityID));
                        var allOwners = GroupMember.MultiGetLocal(allGroups.Select(g => new KeyInfo(g.GroupID, g.OwnerID))).ToDictionary(g => g.GroupID);

                        var memberModels = new List<GroupSearchModel>();
                        foreach (var group in allGroups)
                        {
                            count++;

                            if (!group.IsRootGroup || group.Type != GroupType.Large)
                            {
                                continue;
                            }

                            memberModels.Add(GroupSearchModel.FromData(group, allSettings.GetValueOrDefault(group.GroupID) ?? defaultSettings,
                                allOwners.GetValueOrDefault(group.GroupID), allAvatars.GetValueOrDefault(group.GroupID), allCovers.GetValueOrDefault(group.GroupID)));
                        }

                        if (memberModels.Count > 0)
                        {
                            indexed += memberModels.Count;
                            BulkIndex(client, memberModels);
                        }

                        Console.Write("\rIndexed {0} groups. Processed {1} total groups", indexed, count);
                    }
                    catch (Exception ex)
                    {
                        throw new AerospikeException.ScanTerminated(ex);
                    }
                });
            }
            catch (Exception ex)
            {
                caught = ex;
            }
            sw.Stop();

            Console.WriteLine();
            Console.WriteLine("Finished in {0}", sw.Elapsed);

            if (caught != null)
            {
                Console.WriteLine("Error occurred while indexing groups! Press Enter to see the exception info");
                Console.ReadLine();

                Console.WriteLine("Outer exception:");
                Console.WriteLine(caught.Message);
                Console.WriteLine(caught.StackTrace);

                if (caught.InnerException != null)
                {
                    Console.WriteLine();
                    Console.WriteLine("Inner Exception:");
                    Console.WriteLine(caught.InnerException.Message);
                    Console.WriteLine(caught.InnerException.StackTrace);
                }
            }

            Console.WriteLine("Press Enter to continue");
            Console.ReadLine();
        }

        private static void BulkIndex(ElasticClient client, IEnumerable<GroupSearchModel> members)
        {
            var bulkRequest = new BulkRequest
            {
                Index = GroupSearchManager.IndexName,
                Operations = members.Select(m => (IBulkOperation)new BulkIndexOperation<GroupSearchModel>(m)
                {
                    Routing = m.GroupID.ToString(),
                }).ToArray()
            };
            client.Bulk(bulkRequest);
        }
    }
}
