﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using Aerospike.Client;
using Curse.CloudSearch;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Search;
using Curse.Friends.Enums;
using Nest;

namespace Curse.Friends.ElasticIndexer
{
    class GroupMemberIndexer
    {
        public static void Index(ConfigurationRegion region)
        {
            Console.Clear();

            var aerospikeConfig = GroupMember.GetConfiguration(region.ID);
            var elasticConfigs = SearchConfiguration.Configurations.Where(c => c.RegionIdentifier == region.ID).ToArray();

            var client = new ElasticClient(GroupMemberManager.GetDefaultConnectionSettings(elasticConfigs));
            

            Console.WriteLine("Do you want to delete the index, before re-indexing? [Y|N]");

            var sw = new Stopwatch();
            if (Console.ReadKey(true).Key == ConsoleKey.Y)
            {                
                // Replace template and delete index
                Console.Write("Replacing Group Member template and deleting index...");
                sw.Start();                
                var manager = new GroupMemberManager();
                manager.DeleteTemplate();
                manager.SetupTemplate();
                client.DeleteIndex(GroupMemberManager.IndexName);
                sw.Stop();
                Console.WriteLine("Done in {0}!", sw.Elapsed);
            }

            // Reindex
            Console.WriteLine("Indexing group members...");
            Exception caught = null;
            sw.Restart();
            var indexed = 0;
            var count = 0;
            var serverCount = 0;
            try
            {
                Group.BatchOperate(aerospikeConfig, 1000, groups =>
                {
                    try
                    {
                        var memberModels = new List<GroupMemberSearchModel>();
                        foreach (var group in groups)
                        {
                            count++;

                            if (!group.IsRootGroup || group.Type != GroupType.Large)
                            {
                                continue;
                            }

                            serverCount++;

                            foreach (var member in group.GetAllMembers(true))
                            {
                                memberModels.Add(GroupMemberSearchModel.FromGroupMember(member));

                                if (memberModels.Count >= 1000)
                                {
                                    indexed += memberModels.Count;
                                    BulkIndex(client, memberModels);
                                    memberModels.Clear();
                                }
                            }

                        }

                        if (memberModels.Count > 0)
                        {
                            indexed += memberModels.Count;
                            BulkIndex(client, memberModels);
                        }

                        Console.Write("\rIndexed {0} members. Found {1} servers in {2} total groups", indexed, serverCount, count);
                    }
                    catch (Exception ex)
                    {
                        throw new AerospikeException.ScanTerminated(ex);
                    }
                });
            }
            catch (Exception ex)
            {
                caught = ex;
            }
            sw.Stop();

            Console.WriteLine();
            Console.WriteLine("Finished in {0}", sw.Elapsed);

            if (caught != null)
            {
                Console.WriteLine("Error occurred while indexing group members! Press Enter to see the exception info");
                Console.ReadLine();

                Console.WriteLine("Outer exception:");
                Console.WriteLine(caught.Message);
                Console.WriteLine(caught.StackTrace);

                if (caught.InnerException != null)
                {
                    Console.WriteLine();
                    Console.WriteLine("Inner Exception:");
                    Console.WriteLine(caught.InnerException.Message);
                    Console.WriteLine(caught.InnerException.StackTrace);
                }
            }

            Console.WriteLine("Press Enter to continue");
            Console.ReadLine();
        }

        private static void BulkIndex(ElasticClient client, IEnumerable<GroupMemberSearchModel> members)
        {
            var bulkRequest = new BulkRequest
            {
                Index = GroupMemberManager.IndexName,
                Operations = members.Select(m => (IBulkOperation)new BulkIndexOperation<GroupMemberSearchModel>(m)
                {
                    Routing = m.GroupID.ToString(),
                }).ToArray()
            };
            client.Bulk(bulkRequest);
        }
    }
}
