﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using Curse.Aerospike;
using Curse.CloudSearch;
using Curse.Extensions;
using Curse.Friends.Data;
using Curse.Friends.Data.Messaging;
using Curse.Friends.Enums;
using Elasticsearch.Net;

namespace Curse.Friends.StatsTool.Models
{
    public class MessageCountStats
    {
        public static MessageCountData[] GetMessageCountStats(DateTime startTime, DateTime endTime, TimeSpan interval)
        {
            var allMessages = new List<MessageCountData>();

            var current = startTime;
            var next = startTime + interval;

            var groupCache = new Dictionary<string, Group>();
            while (current < endTime)
            {
                Console.WriteLine("Getting messages for {0}", current.ToString("G"));
                var messages = GetData(current, next, ref groupCache);
                allMessages.AddRange(messages);
                current = next;
                next += interval;
            }

            return allMessages.ToArray();
        }

        private static MessageCountData[] GetData(DateTime current, DateTime next, ref Dictionary<string, Group> groupCache)
        {
            var client = ConversationManager.GetClient();

            var months = TimeSeriesIndexing.GetMonthsBetween(current, next);
            var results = client.Search<ConversationMessage>(s => s
                .Indices(months.Select(d => TimeSeriesIndexing.GetIndexName(ConversationManager.IndexTypeName, d.Year, d.Month)))
                .SearchType(SearchType.Count)
                .Query(f => f.Range(r => r.OnField(t => t.Timestamp).GreaterOrEquals(current.ToEpochMilliseconds()).Lower(next.ToEpochMilliseconds())) &&
                            f.Term(t => t.ConversationType, (int) ConversationType.Group))
                .Aggregations(a => a
                    .Terms("t", ta => ta.Field(t => t.RootConversationID).Size(0)))
                );

            var res = new List<MessageCountData>();
            foreach (var docs in results.Aggs.Terms("t").Items.InSetsOf(1000))
            {
                var missingGroups = new HashSet<Guid>();
                foreach (var doc in docs)
                {
                    if (!groupCache.ContainsKey(doc.Key))
                    {
                        missingGroups.Add(new Guid(doc.Key));
                    }
                }
                if (missingGroups.Count > 0)
                {
                    var groups = Group.MultiGetLocal(missingGroups.Select(s => new KeyInfo(s)));
                    foreach (var group in groups)
                    {
                        groupCache.Add(group.GroupID.ToString(), group);
                    }
                }

                foreach (var doc in docs)
                {
                    Group group;
                    if (!groupCache.TryGetValue(doc.Key, out group))
                    {
                        continue;
                    }

                    if (!group.IsRootGroup)
                    {
                        continue;
                    }

                    if (group.Type != GroupType.Large)
                    {
                        continue;
                    }

                    res.Add(new MessageCountData
                    {
                        StartTime = current,
                        EndTime = next,
                        MessageCount = doc.DocCount,
                        ConversationID = doc.Key,
                        GroupTitle = group.Title
                    });
                }
            }

            return res.ToArray();
        }

        public static void WriteMessageCountStats(string basePath)
        {
            Console.Clear();
            var startTime = new DateTime(2016, 4, 26, 0, 0, 0, DateTimeKind.Utc);
            var endTime = DateTime.UtcNow.Date;
            var interval = TimeSpan.FromDays(1);

            Console.WriteLine("Getting message counts for servers");
            var messages = GetMessageCountStats(startTime, endTime, interval);

            //const string filename = @"C:\temp\message-count-stats.csv";
            var date = DateTime.Now.Date.ToString("yyyy-MM-dd");
            var filename = "message-count-stats.csv";
            var filePath = string.Format(@"{0}\{1}\{2}", basePath, date, filename);
            Directory.CreateDirectory(string.Format(@"{0}\{1}", basePath, date));

            Console.WriteLine("Writing counts to {0}", filePath);

            var sb = new StringBuilder().AppendLine("Start Timestamp,End Timestamp,Group ID,Group Title,Messages");
            foreach (var messageCount in messages)
            {
                sb.AppendLine(string.Format("{0},{1},{2},{3},{4}", messageCount.StartTime.ToString("G"), messageCount.EndTime.ToString("G"), messageCount.ConversationID, messageCount.GroupTitle, messageCount.MessageCount));
            }

            File.WriteAllText(filePath, sb.ToString());
        }
    }
}
