﻿using System;
using System.Collections.Concurrent;
using System.IO;
using System.Linq;
using System.Text;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.StatsTool.Models;

namespace Curse.Friends.StatsTool.Polls
{
    public class PollStats
    {
        public static PollData[] GetPolls()
        {
            var data = new ConcurrentBag<PollData>();

            var count = 0;
            GroupPoll.BatchOperateLocal(500, polls =>
            {
                var pollsArray = polls.ToArray();
                var newPolls = pollsArray.Where(p => p.OptionIDs != null).ToArray();
                var oldPolls = pollsArray.Where(p => p.OptionIDs == null && p.Votes != null);

                foreach (var oldPoll in oldPolls)
                {
                    var pollData = CreatePollData(oldPoll, oldPoll.Votes.Aggregate(0, (curr, kvp) => curr + kvp.Value));
                    data.Add(pollData);
                }

                var allNewOptions = GroupPollOption.MultiGetLocal(newPolls.SelectMany(p => p.OptionIDs.Select(o => new KeyInfo(p.GroupID, p.PollID, o)))).ToLookup(o => o.GroupID.ToString() + o.PollID);
                foreach (var newPoll in newPolls)
                {
                    var pollData = CreatePollData(newPoll, allNewOptions[newPoll.GroupID.ToString() + newPoll.PollID].Aggregate(0, (curr, opt) => curr + opt.VoteCount));
                    data.Add(pollData);
                }

                count += pollsArray.Length;
                Console.Write("\rProcessed {0} polls", count);
            });

            Console.WriteLine();
            return data.ToArray();
        }

        private static PollData CreatePollData(GroupPoll poll, int voteCount)
        {
            return new PollData
            {
                GroupID = poll.GroupID,
                PollID = poll.PollID,
                Title = poll.Title,
                CreationTimestamp = poll.StartDate,
                TotalVoteCount = voteCount
            };
        }

        public static void WritePollStats(string basePath)
        {
            Console.Clear();
            var date = DateTime.Now.Date.ToString("yyyy-MM-dd");
            var filename = "poll-stats.csv";
            var filePath = string.Format(@"{0}\{1}\{2}", basePath, date, filename);

            Directory.CreateDirectory(string.Format(@"{0}\{1}", basePath, date));

            Console.WriteLine("Getting poll info");
            var polls = GetPolls();

            Console.WriteLine("Writing poll info to {0}", filePath);
            var sb = new StringBuilder().AppendLine("Group ID,Poll ID,Title,Created,Votes");
            foreach (var poll in polls)
            {
                sb.AppendLine(string.Format("{0},{1},{2},{3},{4}", poll.GroupID, poll.PollID, poll.Title, poll.CreationTimestamp.ToString("G"), poll.TotalVoteCount));
            }

            File.WriteAllText(filePath, sb.ToString());
        }
    }
}
