﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Configuration;
using Curse.Friends.StatsTool.ExternalCommunity;
using Curse.Friends.StatsTool.Giveaways;
using Curse.Friends.StatsTool.Groups;
using Curse.Friends.StatsTool.Models;
using Curse.Friends.StatsTool.Polls;

namespace Curse.Friends.StatsTool
{
    class Program
    {

        static void Main(string[] args)
        {
            const string defaultBasePath = @"\\fs01a-sna\CurseVoice\AnalyticsDailyStats";

            var auto = args.Any(arg => string.Equals(arg, "auto", StringComparison.OrdinalIgnoreCase));
            var basePath = args.Where(arg => arg.StartsWith("basePath=") && arg.Length > 9).Select(arg => arg.Substring(9)).FirstOrDefault() ?? defaultBasePath;

            Console.Title = "Stats Tool";

            var mode = auto?ConfigurationMode.Release : PickMode();
            StorageConfiguration.Initialize("StatsTool", ConfigurationRegion.USEast.Key, mode, ConfigurationServices.All | ConfigurationServices.LocalOnly);

            Console.Title = "Stats Tool - " + mode;

            if (auto)
            {
                DoEverything(basePath);
            }
            else
            {
                Run(basePath);
            }

            StorageConfiguration.Shutdown();
        }

        private static ConfigurationMode PickMode()
        {
            while (true)
            {
                Console.Clear();
                Console.WriteLine("Which environment do you want to use?");
                foreach (var value in Enum.GetValues(typeof (ConfigurationMode)))
                {
                    Console.WriteLine(value);
                }

                var input = Console.ReadLine();

                ConfigurationMode mode;
                if (Enum.TryParse(input, out mode))
                {
                    return mode;
                }
            }
        }

        private static void Run(string basePath)
        {
            var options = new Dictionary<string, Tuple<string, Action<string>>>
            {
                {"0", new Tuple<string, Action<string>>("Write All", DoEverything)},
                {"1", new Tuple<string, Action<string>>("Write Group Stats", GroupStats.WriteGroupStats)},
                {"2", new Tuple<string, Action<string>>("Write Poll Stats", PollStats.WritePollStats)},
                {"3", new Tuple<string, Action<string>>("Write External Community Mapping Stats", ExternalAccountMappingStats.WriteExternalAccountMappingStats)},
                {"4", new Tuple<string, Action<string>>("Write Message Count Stats", MessageCountStats.WriteMessageCountStats)},
                {"5", new Tuple<string, Action<string>>("Write Giveaway Stats", GiveawayStats.WriteGiveawayStats)}
            };

            while (true)
            {
                Console.Clear();
                foreach (var option in options)
                {
                    Console.WriteLine("{0} - {1}", option.Key, option.Value.Item1);
                }
                Console.WriteLine("exit - Exit the application");

                var input = Console.ReadLine();

                if(string.Equals(input, "exit", StringComparison.InvariantCultureIgnoreCase))
                {
                        return;
                }

                Tuple<string, Action<string>> selected;
                if (input!=null && options.TryGetValue(input, out selected))
                {
                    selected.Item2(basePath);
                }
                else
                {
                    Console.WriteLine("Invalid option selected");
                }

                Console.WriteLine("Press enter to continue");
                Console.ReadLine();
            }
        }

        private static void DoEverything(string basePath)
        {
            GroupStats.WriteGroupStats(basePath);
            PollStats.WritePollStats(basePath);
            ExternalAccountMappingStats.WriteExternalAccountMappingStats(basePath);
            MessageCountStats.WriteMessageCountStats(basePath);
            GiveawayStats.WriteGiveawayStats(basePath);
        }
    }
}
