﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using Curse.Aerospike;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.VaultSharp;

namespace Curse.Tools.ClientEndpointFixer
{
    class Program
    {
        private static readonly Regex _serverNameRegex = new Regex(@"^[a-zA-Z]+[0-9]+[aAbBcC]?-[a-zA-Z]+$", RegexOptions.Compiled);

        static void Main(string[] args)
        {
            if (args.Length < 1)
            {
                Console.WriteLine("Usage: epfixer <machine-name> <should-resolve>");
                return;
            }

            var serverName = args[0].ToUpper();
            if (!_serverNameRegex.IsMatch(serverName))
            {
                Console.WriteLine("Server name doesn't appear to be valid");
                return;
            }            
#if DEBUG
            StorageConfiguration.Initialize("ClientEndpointFixer", ConfigurationRegion.USEast.Key, ConfigurationMode.Debug, ConfigurationServices.Database | ConfigurationServices.Queue);
#else
            VaultSharpHelper.ReplaceSecrets(AppDomain.CurrentDomain.BaseDirectory, "release");
            StorageConfiguration.Initialize("ClientEndpointFixer", null, null, ConfigurationServices.Database | ConfigurationServices.Queue | ConfigurationServices.LocalOnly);
#endif

            Console.WriteLine("Fix endpoints in Region {0} for Server {1}?", StorageConfiguration.CurrentRegion.Key, serverName);
            if (Console.ReadKey(true).Key != ConsoleKey.Y)
            {
                return;
            }

            var totalCount = 0;
            var updatedCount = 0;
            var failedCount = 0;

            ClientEndpoint.BatchOperateOnIndexLocal(ep => ep.ServerName, serverName, 1000, eps =>
            {
                var updated = new List<ClientEndpoint>();

                foreach (var ep in eps)
                {
                    if (!ep.ServerName.Equals(serverName))
                    {
                        ++failedCount;
                        continue;
                    }

                    ep.DisconnectedDate = DateTime.UtcNow;
                    ep.IsConnected = false;                    
                    ep.ServerName = string.Empty;

                    try
                    {
                        ep.Update(UpdateMode.Concurrent, p => p.DisconnectedDate, p => p.IsConnected, p => p.ServerName);
                        updated.Add(ep);

                    }
                    catch (Exception)
                    {
                        ++failedCount;
                        // Ignore
                    }

                    totalCount++;
                }

                if (updated.Count > 0)
                {
                    var allUsers = UserRegion.MultiGetLocal(new HashSet<int>(updated.Select(s => s.UserID)).Select(id => new KeyInfo(id)));
                    foreach (var userRegion in allUsers)
                    {
                        ClientEndpointResolver.Create(userRegion.UserID, userRegion.RegionID);
                    }
                }

                updatedCount += updated.Count;
                Console.WriteLine("{0}/{1} endpoints updated ({2} skipped or failed)", updatedCount.ToString("N0"), totalCount.ToString("N0"), failedCount.ToString("N0"));
            });

            Console.WriteLine("Finished fixing endpoints in Region {0} for Server {1}", StorageConfiguration.CurrentRegion.Key, serverName);
            Console.WriteLine("Press Enter to exit");
            Console.ReadLine();

            StorageConfiguration.Shutdown();
        }
    }
}
