﻿using CsvHelper;
using Curse.Friends.Configuration;
using Curse.Friends.Data.Models;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows.Forms;

namespace TwitchGameComparison
{
    public static class GenerateComparison
    {
        private static List<GameRecord> twitchRecords = new List<GameRecord>();
        private static List<GameRecord> curseRecords = new List<GameRecord>();        

        public static void Execute(bool excludeItemsInAerospike)
        {
            Console.WriteLine("Open Twitch CSV");

            var dlg = new OpenFileDialog();
            dlg.Filter = "CSV|*.csv";

            var twitchFile = string.Empty;
            var curseFile = string.Empty;
            dlg.Title = "Twitch CSV";
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                twitchFile = dlg.FileName;
            }
            else
            {
                Console.WriteLine("No twitch file selected.");
                return;
            }

            Console.WriteLine("Open Curse CSV");
            dlg.Title = "Curse CSV";
            if (dlg.ShowDialog() == DialogResult.OK)
            {
                curseFile = dlg.FileName;
            }
            else
            {
                Console.WriteLine("No curse file selected.");
                return;
            }

            if (!string.IsNullOrWhiteSpace(curseFile) && !string.IsNullOrWhiteSpace(twitchFile))
            {
                using (TextReader reader = File.OpenText(twitchFile))
                {
                    var csv = new CsvReader(reader);
                    while (csv.Read())
                    {
                        twitchRecords.Add(new GameRecord
                        {
                            Name = csv.GetField(0),
                            ID = csv.GetField<int>(1)
                        });
                    }
                }

                using (TextReader reader = File.OpenText(curseFile))
                {
                    var csv = new CsvReader(reader);
                    while (csv.Read())
                    {
                        curseRecords.Add(new GameRecord
                        {
                            ID = csv.GetField<int>("ID"),
                            Name = csv.GetField<string>("Name")
                        });
                    }
                }
            }

            if (twitchRecords.Count == 0)
            {
                Console.WriteLine("No twitch records imported.");
                Console.ReadLine();
                return;
            }

            if (curseRecords.Count == 0)
            {
                Console.WriteLine("No curse records imported");
                Console.ReadLine();
                return;
            }

            if(excludeItemsInAerospike)
            {                                
                var savedItems = TwitchGameMapping.GetAllLocal();
                int excluded = curseRecords.RemoveAll(x => savedItems.Any(y => y.CurseGameID == x.ID));
                Console.WriteLine("# of records already in aerospike: {0}", excluded);
            }

            var outputFolder = string.Empty;
            var folderBrowser = new FolderBrowserDialog();
            if (folderBrowser.ShowDialog() == DialogResult.OK)
            {
                outputFolder = folderBrowser.SelectedPath;
            }
            else
            {
                Console.WriteLine("No folder selected");
                return;
            }


            var matchFile = Path.Combine(outputFolder, "matches.csv");
            var diffFile = Path.Combine(outputFolder, "differences.csv");

            var matches = new List<MatchGameRecord>();
            var differences = new List<DifferenceGameRecord>();
            foreach (var curseGame in curseRecords)
            {
                var twitchGame = twitchRecords.FirstOrDefault(x => string.Compare(curseGame.Name, x.Name, true) == 0);
                if (twitchGame != null)
                {
                    matches.Add(new MatchGameRecord { CurseID = curseGame.ID, TwitchID = twitchGame.ID, Name = twitchGame.Name });
                }
                else
                {
                    differences.Add(new DifferenceGameRecord { CurseID = curseGame.ID, CurseName = curseGame.Name });
                }
            }

            matches.AddRange(differences.Select(x => new MatchGameRecord { CurseID = x.CurseID, Name = x.CurseName }));

            if (matches.Any())
            {
                using (TextWriter textWriter = File.CreateText(matchFile))
                {
                    var writer = new CsvWriter(textWriter);
                    writer.WriteHeader<MatchGameRecord>();
                    writer.WriteRecords(matches);
                }

                Console.WriteLine(string.Format("number of matching names: {0}.", matches.Count));
            }

            if (differences.Any())
            {
                using (TextWriter textWriter = File.CreateText(diffFile))
                {
                    var writer = new CsvWriter(textWriter);
                    writer.WriteHeader<DifferenceGameRecord>();
                    writer.WriteRecords(differences);
                }

                Console.WriteLine(string.Format("number of different names: {0}.", differences.Count));
            }

            Process.Start(outputFolder);
        }

    }
}
