﻿import { DateUtils } from './DateUtils';

export abstract class BaseWebServiceClient {
    _url: string;
    _getToken: () => string;
    _withCredentials: boolean;
    _onAuthorizeFailed: () => void;

    constructor(url: string, getToken?: () => string, withCredentials?: boolean, onAuthorizeFailed: () => void) {
        this._url = url;
        this._getToken = getToken;
        this._withCredentials = withCredentials;
    }

    protected _get<T>(action: string): Promise<T> {
        return this._request<T>('GET', action);
    }

    protected _post<T>(action: string, contract?: any): Promise<T> {
        return this._request<T>('POST', action, contract);
    }

    protected _patch<T>(action: string, contract?: any): Promise<T> {
        return this._request<T>('PATCH', action, contract);
    }

    protected _delete<T>(action: string, contract?: any): Promise<T> {
        return this._request<T>('DELETE', action, contract);
    }

    protected _options<T>(action: string, contract?: any): Promise<T> {
        return this._request<T>('OPTIONS', action, contract);
    }

    private _request<T>(verb: string, action: string, contract?: any): Promise<T> {
        const url = action ? this._url + '/' + action : this._url;
        const headers = {};

        const token = this._getToken && this._getToken();
        if (token) {
            headers['AuthenticationToken'] = token;
        }

        return new Promise<T>((resolve, reject) => {
            const req = new XMLHttpRequest();
            req.withCredentials = this._withCredentials;

            req.onload = () => {
                if (req.status === 401) {
                    this._onAuthorizeFailed();
                    return;
                }

                if (req.status >= 200 && req.status < 300) {
                    if (req.status === 204 || !req.responseText) {
                        resolve();
                        return;
                    }

                    resolve(JSON.parse(req.responseText, (k, v) => DateUtils.tryParseDate(v)));
                    return;
                }
                

                if (req.responseText) {
                    try {
                        reject(JSON.parse(req.responseText));
                        return;
                    } catch (e) {}
                } else if (req.statusText) {
                    reject(Error(req.statusText));
                }

                reject();
            };

            req.onerror = error => {                
                reject(error);
            };

            req.open(verb, url);

            req.setRequestHeader('Content-type', 'application/json; charset=utf-8');
            for (const header in headers) {
                req.setRequestHeader(header, headers[header]);
            }

            if (contract) {
                req.send(JSON.stringify(contract));
            } else {
                req.send();
            }
        });
    }
}
