﻿import { DateUtils } from './DateUtils';
import { Guid } from './Guid';

interface Message {
    TypeID: number;
    Body: any;
}

export interface ContractMap {
    [key: number]: (contract: any) => void;
}

export abstract class BaseWebSocketClient {
    private contractMap: ContractMap = {};
    private ws: WebSocket;

    constructor(url: string, contractMap: ContractMap) {
        this.contractMap = contractMap;

        const id = Guid.newGuid();

        this.ws = new WebSocket(url);
        this.ws.onopen = e => this.onSocketOpen(id);
        this.ws.onclose = e => this.onSocketClose(id, e);
        this.ws.onmessage = e => this.receiveMessage(e.data);
    }

    close() {
        this.ws.close();
    }

    sendContract(typeID: number, contract: any) {
        const message: Message = { TypeID: typeID, Body: contract };
        this.ws.send(JSON.stringify(message));
    }

    receiveMessage(val: string) {
        const message: Message = JSON.parse(val, (k, v) => DateUtils.tryParseDate(v));
        const method = this.contractMap[message.TypeID];

        if (method !== null) {
            method(message.Body);
        }
    }

    // Event handlers
    protected onSocketOpen(id: string) { }
    protected onSocketClose(id: string, e: CloseEvent) { }
}
