#Globals
$ScriptPath = Split-Path $MyInvocation.MyCommand.Path
$Git = "$ScriptPath\Git\cmd\git.exe"
$MsBuild = 'C:\Windows\Microsoft.NET\Framework64\v4.0.30319\MSBuild.exe'
$GitUser = "teamcity"
$BaseRepoUrl = "https://$GitUser@github.curse.us"

Start-Transcript -Path "$ScriptPath\LatestLog.txt"

$config = ([xml](Get-Content "$ScriptPath\RunLoadTest.config")).FirstChild

#Functions
$MessageNumber = 1
function Log-Message
{
    param
    (
        [Parameter(Mandatory=$True, Position=1)]
        [string]$message,

        [Parameter(Position=2)]
        [string]$entryType='Information',

        [int]$eventID=$Script:MessageNumber,

        [byte[]]$rawData,

        [string]$logName='Application',
        [string]$source='RunLoadTestScript'
    )

    if(! [System.Diagnostics.EventLog]::SourceExists($source))
    {
        New-EventLog -LogName $logName -Source $source
    }

    if($rawData)
    {
        Write-EventLog -LogName $logName -Source $source -EntryType $entryType -Message $message -RawData $rawData -EventId $eventID
    }
    else
    {
        Write-EventLog -LogName $logName -Source $source -EntryType $entryType -Message $message -EventId $eventID
    }

    if($entryType -eq "Error")
    {
        Write-Error $message
    }
    elseif($entryType -eq "Warning")
    {
        Write-Warning $message
    }
    else
    {
        Write-Host $message
    }

    $Script:MessageNumber=$Script:MessageNumber+1
}

function Test-Repo
{
    param
    (
        [Parameter(Position=1)]
        [string]$repoPath
    )

    # Simple path check for now - want to add some checking to make sure the repo is not screwed up
    Test-Path $repoPath
}

function Update-Repo
{
    param
    (
        [string]$RepoPath,
        
        [string]$Branch="master"
    )

    cd $RepoPath

    # Clean working state
    & $Script:Git reset --hard -q *>&1 | Out-Host
    & $Script:Git clean -qfdx *>&1 | Out-Host

    # Determine if an update is needed
    & $Script:Git fetch origin
    $currentBranch = & $Script:Git rev-parse --abbrev-ref HEAD
    if($Branch -ne $currentBranch)
    {
        Log-Message "Switching to branch $Branch in $RepoPath"
        $wasUpdated = $true
        & $Script:Git checkout "$Branch" -q *>&1 | Out-Host
    }

    # Update/Change Branch
    $localrev = & $Script:Git rev-parse '@'
    $remoterev = & $Script:Git rev-parse '@{u}'

    if(!$localrev.Equals($remoterev))
    {
        Log-Message "Updating $RepoPath"
        $wasUpdated = $true
        & $Script:Git pull -q *>&1 | Out-Host
    }

    cd $Script:ScriptPath

    return $wasUpdated
}

function Get-RepoPath
{
    param
    (
        [Parameter(Position=1)]
        $repoName
    )

    "Work\$repoName\Repository"
}

$Env:GIT_TERMINAL_PROMPT = "0"

robocopy "\\smb02a-hsv.curse.us\Files\LoadTesting" $ScriptPath /E /XO /XF README.txt /NP /NFL /NDL /NJH /NJS

if(!(Test-Path $Git))
{
    Log-Message "Could not find Git at $Git" Error
    exit 404 # lawl
}

# Checkout

foreach($repo in $config.Repositories.ChildNodes | where {$_.NodeType -ne 'Comment'})
{
    $repoPath = Get-RepoPath $repo.Name

    # Ensure the repo exists and has been checked out
    $exists = Test-Repo $repoPath
    if(!($exists))
    {
        Log-Message "Repo $($repo.Name) does not exist, cloning it into $repoPath."

        if($repo.URL.Contains("://"))
        {
            # Absolute URL, assume it does not live in our repos
            & $Git clone -q $repo.URL $repoPath *>&1 | Out-Host
        }
        else
        {
            $credentialBase = "credential.https://github.curse.us"
            $credentialPath = "!$ScriptPath\git-credential-winstore.exe" -replace "\\","/"
            & $Git clone -q "$BaseRepoUrl/$($repo.URL)" $repoPath -c "$credentialBase.helper=$credentialPath" -c "$credentialBase.username=$GitUser" *>&1 | Out-Host
        }

        if($LASTEXITCODE -ne 0)
        {
            Log-Message "Git Clone returned $LASTEXITCODE. Terminating" Error
            exit $LASTEXITCODE
        }
    }

    Update-Repo -RepoPath $repoPath -Branch $repo.Branch | Out-Null
}

#Deploy
foreach($deploy in $config.Deploys.ChildNodes | where {$_.NodeType -ne 'Comment'})
{
    Log-Message "Deploying $($deploy.Name)"

    $projPath = "$ScriptPath\$(Get-RepoPath $deploy.RepoName)\$($deploy.ProjectFilePath)"
    & $MsBuild '/v:m' "$projPath" *>&1 | Out-Host

    if($LASTEXITCODE -ne 0)
    {
        Log-Message "Deploy returned $LASTEXITCODE. Terminating" Error
        exit $LASTEXITCODE
    }
}

# Build
Log-Message 'Building LoadTesting' 

$LoadTestingDirectory = "$ScriptPath\$(Get-RepoPath "CurseVoiceLoadTesting")"
& $MsBuild '/v:m' '/property:Configuration=ConsoleApp' '/property:Platform=Any CPU' "$LoadTestingDirectory\LoadTesting.sln" *>&1 | Out-Host

if($LASTEXITCODE -ne 0)
{
    Log-Message "MSBuild returned $LASTEXITCODE. Terminating" Error
    exit $LASTEXITCODE
}

# Force Load Web Services to hopefully reduce timeouts during site activation

foreach($siteToLoad in $config.WebSites.ChildNodes  | where {$_.NodeType -ne 'Comment'})
{
    Invoke-WebRequest -Uri $siteToLoad.URL -UseBasicParsing
}

# Run
Log-Message 'Running Load Test'

cd "$LoadTestingDirectory\Source\Curse.LoadTests.Coordinator\bin\ConsoleApp"
& '.\CurseLoadTestsCoordinator.exe' *>&1 | Out-Host
cd $ScriptPath

if($LASTEXITCODE -ne 0)
{
    Log-Message "Coordinator returned non-zero exit code $LASTEXITCODE" Error
    Exit $LASTEXITCODE
}

Log-Message 'Finished load test!'

Stop-Transcript
