﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.LoadTests.Client;
using Curse.LoadTests.Client.Behavior;
using Curse.LoadTests.Client.Models;
using Curse.LoadTests.Enums;
using Curse.Logging;

namespace Curse.ClientBehavior.Behavior
{
    internal static class Actions
    {
        private static readonly Dictionary<int, ClientAction> ActionsDictionary = new Dictionary<int, ClientAction>();
        private static readonly int MaxRand;

        static Actions()
        {
            var actions = new[]
            {
                // General Actions
                new ClientAction(DoNothing, 3000),
                new ClientAction(ChangeProfile, 10, TimeSpan.FromMinutes(75)),
                new ClientAction(ChangeGameStatus, 100, TimeSpan.FromMinutes(10)),
                new ClientAction(ChangeConnectionStatus, 10),
                new ClientAction(HangUp, 100),
                new ClientAction(Logout, 0, TimeSpan.FromMinutes(60)),

                // Friend-Related Actions
                new ClientAction(AddFriend, 50, TimeSpan.FromMinutes(1)),
                new ClientAction(MessageFriend, 500),
                new ClientAction(RemoveFriend, 5, TimeSpan.FromMinutes(45)),
                new ClientAction(CallFriend, 100, TimeSpan.FromMinutes(2)),
                new ClientAction(RenameFriend, 10, TimeSpan.FromMinutes(10)),

                // Group-Related Actions
                new ClientAction(CreateNormalGroup, 50, TimeSpan.FromMinutes(30)),
                new ClientAction(CreateClan, 25, TimeSpan.FromMinutes(90)),
                new ClientAction(CreateSubgroup, 5, TimeSpan.FromMinutes(75)),
                new ClientAction(AddUsersToGroup, 50, TimeSpan.FromMinutes(10)),
                new ClientAction(RemoveUsersFromGroup, 0, TimeSpan.FromMinutes(45)),
                new ClientAction(LeaveGroup, 30, TimeSpan.FromMinutes(30)),
                new ClientAction(ChangeGroupInfo, 10, TimeSpan.FromMinutes(30)),
                new ClientAction(ChangeGroupMemberRole, 10, TimeSpan.FromMinutes(30)),
                new ClientAction(MessageGroup, 500),
                new ClientAction(CallGroup, 100, TimeSpan.FromMinutes(2)),
                new ClientAction(DeleteRootGroup, 0, TimeSpan.FromMinutes(90)),
                new ClientAction(DeleteSubgroup, 0, TimeSpan.FromMinutes(75)),
            };

            var score = 0;
            foreach (var action in actions)
            {
                if (action.Weight == 0)
                {
                    // Skip actions with 0 weight entirely
                    continue;
                }
                score += action.Weight;
                ActionsDictionary[score] = action;
            }
            if (score < 1)
            {
                throw new InvalidOperationException("No actions specified.");
            }
            MaxRand = score;
        }

        public static void PerformAction(LoadTests.Client.Behavior.ClientBehavior client, Random random)
        {
            try
            {
                // Always be logged in before trying to perform actions
                if (!client.IsLoggedIn)
                {
                    if (client.Login())
                    {
                        ReloadFriends(client);
                        ReloadGroups(client);
                    }
                    return;
                }

                // If there are friend requests pending, respond to one before performing any actions
                RespondToFriendRequest(client, random);

                // Ensure you have at least one friend - this bypasses any minimum interval there may be
                if (!client.Friends.Any() && random.Next(0,99)<25)
                {
                    AddFriend(client);
                    return;
                }

                // Perform a random action
                ClientAction action = null;
                while (action == null)
                {
                    var rand = random.Next(1, MaxRand);
                    var index = ActionsDictionary.Select(kvp => kvp.Key).DefaultIfEmpty(ActionsDictionary.Last().Key).FirstOrDefault(v => v >= rand);
                    action = ActionsDictionary[index];

                    if (!action.CanPerform(client))
                    {
                        // Can't do this, so choose another
                        action = null;
                    }
                }

                action.Perform(client);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unexpected exception thrown while performing an action.");
            }
        }

        #region Friends

        private static void ReloadFriends(LoadTests.Client.Behavior.ClientBehavior client)
        {
            client.QueueActionOrEvent("GetMyFriends");
            var response = client.FriendsWebClient.Call("GetMyFriends", svc => svc.GetMyFriends());
            client.QueueActionOrEvent(response);

            if (response.Status == GetMyFriendsStatus.Successful)
            {
                foreach (var friend in response.Friends)
                {
                    client.Friends.AddOrUpdate(friend.OtherUserID,
                        id => new Friend(friend),
                        (id, f) => f.UpdateFromFriendship(friend));
                }

                var removedFriends = client.Friends.Values.Where(f => response.Friends.All(fr => fr.OtherUserID != f.UserID));
                foreach (var removedFriend in removedFriends)
                {
                    Friend friend;
                    client.Friends.TryRemove(removedFriend.UserID, out friend);
                }
            }
        }

        private static void AddFriend(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var nonFriend = client.GetRandomNonFriend();
            if (nonFriend == null)
            {
                // no friends to add
                return;
            }

            var findRequest = new FriendSearchRequest {QueryString = nonFriend.Username};
            client.QueueActionOrEvent(findRequest);
            var findResponse = client.FriendsWebClient.Call("FindFriends", svc => svc.FindFriends(findRequest));
            client.QueueActionOrEvent(findResponse);

            var request = new FriendshipRequest
            {
                FriendID = nonFriend.UserID,
                InvitationMessage = "Won't you be my neighbor?",
                IsFromSuggestion = false,
                KnownIdentity = nonFriend.Username
            };

            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("AddFriend", svc => svc.RequestFriendship(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case RequestFriendshipStatus.Successful:
                    LoadTest.Track(TrackedEventType.FriendRequestSent);
                    break;
                case RequestFriendshipStatus.AlreadyRequested:
                    if (!client.FriendsWith(nonFriend.UserID))
                    {
                        Logger.Warn("Failed to send Friend Request", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.FriendRequestError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to send Friend Request", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.FriendRequestError);
                    break;
            }
        }

        private static void MessageFriend(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var randomFriends = client.GetRandomFriends().Where(f => f.FriendshipStatus == FriendshipStatus.Confirmed && f.ConnectionStatus == UserConnectionStatus.Online).ToArray();
            if (randomFriends.Length == 0)
            {
                // no friend to message
                return;
            }

            var friendToMessage = randomFriends.First();
            var instantMessageRequest = new InstantMessageRequest
            {
                FriendID = friendToMessage.UserID,
                ClientID = Guid.NewGuid(),
                Message = string.Format("Thanks for being my neighbor {0}.", friendToMessage.UserID)
            };

            var notificationClient = client.NotificationClient;
            if (notificationClient == null)
            {
                // We've been disconnected!
                return;
            }

            client.QueueActionOrEvent(instantMessageRequest);
            notificationClient.SendInstantMessage(instantMessageRequest.FriendID, instantMessageRequest.Message, instantMessageRequest.ClientID);
        }

        private static void RemoveFriend(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var randomFriends = client.GetRandomFriends().Where(f => f.FriendshipStatus == FriendshipStatus.Confirmed || f.FriendshipStatus == FriendshipStatus.AwaitingThem).ToArray();
            if (randomFriends.Length == 0)
            {
                // no friend to remove
                return;
            }

            var friendToRemove = randomFriends.First();
            var request = new RemoveFriendshipRequest
            {
                FriendID = friendToRemove.UserID
            };
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("RemoveFriend", svc => svc.RemoveFriendship(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case BasicServiceResponseStatus.Successful:
                    LoadTest.Track(TrackedEventType.RemoveFriend);
                    break;
                case BasicServiceResponseStatus.NotFound:
                    if (client.FriendsWith(friendToRemove.UserID))
                    {
                        Logger.Warn("Failed to Remove Friend", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.RemoveFriendError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Remove Friend", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.RemoveFriendError);
                    break;
            }
        }

        private static void CallFriend(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var randomFriends = client.GetRandomFriends().Where(f => f.FriendshipStatus == FriendshipStatus.Confirmed && f.IsOnline).ToArray();
            if (randomFriends.Length == 0)
            {
                // No friend available
                return;
            }
            var friend = randomFriends.First();
            client.VoiceManager.CallFriend(friend.UserID);
        }

        private static void RenameFriend(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var randomFriends = client.GetRandomFriends().Where(f => f.FriendshipStatus == FriendshipStatus.Confirmed && f.Nickname == null).ToArray();
            if (randomFriends.Length == 0)
            {
                // No friends available
                return;
            }

            var friend = randomFriends.First();
            var request = new RenameFriendRequest {FriendID = friend.UserID, Nickname = friend.Username + " Nickname"};
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("RenameFriend", svc => svc.RenameFriend(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case RenameFriendStatus.Successful:
                    LoadTest.Track(TrackedEventType.ChangedFriendInfo);
                    break;
                case RenameFriendStatus.NotFound:
                    if (client.FriendsWith(friend.UserID))
                    {
                        Logger.Warn("Failed to Rename Friend", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.ChangeFriendInfoError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Rename Friend", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.ChangeFriendInfoError);
                    break;
            }
        }

        #endregion

        #region Group

        public static void ReloadGroups(LoadTests.Client.Behavior.ClientBehavior client)
        {
            client.QueueActionOrEvent("GetMyGroups");
            var response = client.FriendsWebClient.Call("GetMyGroups", svc => svc.GetMyGroups());
            client.QueueActionOrEvent(response);

            if (response.Status == BasicServiceResponseStatus.Successful)
            {
                var groupings = response.Groups.GroupBy(g => g.RootGroupID).ToArray();

                // Remove groups not in the response
                var groupingLookup = groupings.ToLookup(g => g.Key);
                foreach (var groupID in client.Groups.Keys.Where(id => !groupingLookup.Contains(id)))
                {
                    Group removed;
                    client.Groups.TryRemove(groupID, out removed);
                }

                // Add or update groups in the response
                foreach (var grouping in groupings)
                {
                    var rootGroup = grouping.FirstOrDefault(g => g.RootGroupID == g.GroupID);
                    if (rootGroup == null)
                    {
                        continue;
                    }

                    var root = client.Groups.AddOrUpdate(grouping.Key, id => new Group(rootGroup), (id, g) => g.UpdateFromGroupMembership(rootGroup));
                    foreach (var groupMembership in grouping)
                    {
                        if (groupMembership.GroupID == groupMembership.RootGroupID)
                        {
                            continue;
                        }
                        root.AddOrUpdateSubgroup(groupMembership);
                    }
                }
            }
        }

        private static void CreateNormalGroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var randomFriends = client.GetRandomFriends(1, 10).Where(f => f.FriendshipStatus == FriendshipStatus.Confirmed).ToArray();
            if (randomFriends.Length == 0)
            {
                // No friends to add
                return;
            }

            var request = new CreateGroupRequest
            {
                Type = GroupType.Normal,
                AvatarUrl = ClientConstants.GroupAvatarUrlDefault,
                Title = string.Format("Normal Group - {0}", client.CurrentUser.UserID),
                RecipientsUserIDs = randomFriends.Select(f => f.UserID).ToArray()
            };
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("CreateGroup", svc => svc.CreateGroup(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case CreateGroupStatus.Successful:
                    LoadTest.Track(TrackedEventType.NormalGroupCreated);
                    break;
                case CreateGroupStatus.InvalidRecipients:
                    // Check for a race condition where we've been unfriended
                    if (request.RecipientsUserIDs.All(id => client.FriendsWith(id, FriendshipStatus.Confirmed)))
                    {
                        Logger.Warn("Failed to Create Normal Group", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.NormalGroupCreationError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Create Normal Group", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.NormalGroupCreationError);
                    break;
            }
        }

        private static void CreateClan(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var randomFriends = client.GetRandomFriends(1, 10).Where(f => f.FriendshipStatus == FriendshipStatus.Confirmed).ToArray();
            if (randomFriends.Length == 0)
            {
                // No friends to add
                return;
            }

            var request = new CreateGroupRequest
            {
                Type = GroupType.Large,
                AvatarUrl = ClientConstants.GroupAvatarUrlDefault,
                Title = string.Format("Clan - {0}", client.CurrentUser.UserID),
                RecipientsUserIDs = randomFriends.Select(f => f.UserID).ToArray(),
                LobbyName = string.Format("Clan Lobby - {0}", client.CurrentUser.UserID),
                MessageOfTheDay = "Winning",
                DefaultRoles = Enum.GetValues(typeof (GroupRole)).Cast<GroupRole>().ToDictionary(r => r, r => r.ToString())
            };

            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("CreateGroup", svc => svc.CreateGroup(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case CreateGroupStatus.Successful:
                    LoadTest.Track(TrackedEventType.ClanCreated);
                    break;
                case CreateGroupStatus.InvalidRecipients:
                    // Check for a race condition where we've been unfriended
                    if (request.RecipientsUserIDs.All(id => client.FriendsWith(id, FriendshipStatus.Confirmed)))
                    {
                        Logger.Warn("Failed to Create Clan", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.ClanCreationError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Create Clan", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.ClanCreationError);
                    break;
            }
        }

        private static void CreateSubgroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var clan = client.GetRandomRootGroup(GroupType.Large, GroupRole.Admin);
            if (clan == null)
            {
                // No clans
                return;
            }

            var request = new CreateGroupRequest
            {
                Type = GroupType.Large,
                Title = string.Format("Clan - {0}", client.CurrentUser.UserID),
                ParentGroupID = clan.GroupID,
                AccessLevel = GroupRole.Officer
            };

            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("CreateGroup", svc => svc.CreateGroup(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case CreateGroupStatus.Successful:
                    LoadTest.Track(TrackedEventType.SubgroupCreated);
                    break;
                case CreateGroupStatus.InvalidRecipients:
                    // Check for a race condition where we've been unfriended
                    if (request.RecipientsUserIDs.All(id => client.FriendsWith(id, FriendshipStatus.Confirmed)))
                    {
                        Logger.Warn("Failed to Create Subgroup", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.SubgroupCreationError);
                    }
                    break;
                case CreateGroupStatus.Forbidden:
                    // Check for a race condition where we've been removed/left/demoted from the clan
                    if (client.MemberOfGroup(clan.GroupID, clan.GroupID, GroupRole.Admin))
                    {
                        Logger.Warn("Failed to Create Subgroup", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.SubgroupCreationError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Create Subgroup", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.SubgroupCreationError);
                    break;
            }
        }

        private static void AddUsersToGroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var group = client.GetRandomRootGroup(null, GroupRole.Officer);
            if (group == null)
            {
                // No group available
                return;
            }

            var friendsToAdd = client.GetRandomFriends(1, 2).Where(f => f.FriendshipStatus == FriendshipStatus.Confirmed && !group.IsMember(f.UserID)).ToArray();
            if (friendsToAdd.Length == 0)
            {
                // No friends to add
                return;
            }


            var request = new AddUserToGroupRequest
            {
                GroupID = group.GroupID,
                UserIDs = friendsToAdd.Select(f => f.UserID).ToArray()
            };

            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("AddUsersToGroup", svc => svc.AddUsersToGroup(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case BasicServiceResponseStatus.Successful:
                    LoadTest.Track(TrackedEventType.AddedGroupMember);
                    break;
                case BasicServiceResponseStatus.NotFound:
                    // Check to ensure the group still exists
                    if (client.MemberOfGroup(group.GroupID, group.RootGroupID, GroupRole.Officer))
                    {
                        Logger.Warn("Failed to Add users to group", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.AddGroupMemberError);
                    }
                    break;
                case BasicServiceResponseStatus.Forbidden:
                    // Check to ensure the user is still a member and the added users are friends
                    if (client.MemberOfGroup(group.GroupID, group.RootGroupID, GroupRole.Officer) &&
                        request.UserIDs.All(id => client.FriendsWith(id, FriendshipStatus.Confirmed)))
                    {
                        Logger.Warn("Failed to Add users to group", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.AddGroupMemberError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Add users to group", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.AddGroupMemberError);
                    break;
            }
        }

        private static void RemoveUsersFromGroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var group = client.GetRandomRootGroup(null, GroupRole.Admin);
            if (group == null)
            {
                // No group available
                return;
            }

            var memberToRemove = group.Members.FirstOrDefault(m => m.UserID != client.CurrentUser.UserID);
            if (memberToRemove == null)
            {
                // No member to remove
                return;
            }

            var request = new RemoveUsersFromGroupRequest
            {
                GroupID = group.GroupID,
                UserIDs = new[] {memberToRemove.UserID}
            };

            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("RemoveUsersFromGroup", svc => svc.RemoveUsersFromGroup(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case BasicServiceResponseStatus.Successful:
                    LoadTest.Track(TrackedEventType.RemovedGroupMember);
                    break;
                case BasicServiceResponseStatus.NotFound:
                case BasicServiceResponseStatus.Forbidden:
                    if (client.MemberOfGroup(group.GroupID, group.RootGroupID, GroupRole.Admin))
                    {
                        Logger.Warn("Failed to Remove users from group", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.RemoveGroupMemberError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Remove users from group", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.RemoveGroupMemberError);
                    break;
            }
        }

        private static void LeaveGroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var group = client.GetRandomRootGroup();
            if (group == null)
            {
                // No group available
                return;
            }

            var request = new LeaveGroupRequest
            {
                GroupID = group.GroupID
            };

            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("LeaveGroup", svc => svc.LeaveGroup(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case BasicServiceResponseStatus.Successful:
                    LoadTest.Track(TrackedEventType.LeftGroup);
                    break;
                case BasicServiceResponseStatus.NotFound:
                case BasicServiceResponseStatus.Forbidden:
                    if (client.MemberOfGroup(group.GroupID, group.RootGroupID))
                    {
                        Logger.Warn("Failed to Leave group", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.LeaveGroupError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Leave group", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.LeaveGroupError);
                    break;
            }
        }

        private static void MessageGroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var group = client.GetRandomGroup();
            if (group == null)
            {
                // No group available
                return;
            }

            var notificationClient = client.NotificationClient;
            if (notificationClient == null)
            {
                return;
            }

            var request = new GroupMessageRequest
            {
                ClientID = Guid.NewGuid(),
                GroupID = group.GroupID,
                Message = string.Format("Hello, Neighborhood - Much Love, {0}!", client.CurrentUser.UserID)
            };
            client.QueueActionOrEvent(request);
            notificationClient.SendGroupMessage(request.GroupID, request.Message, request.ClientID);
        }

        private static void CallGroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var group = client.GetRandomGroup();
            if (group == null)
            {
                // No group available
                return;
            }

            client.VoiceManager.CallGroup(group.GroupID);
        }

        private static void ChangeGroupInfo(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var group = client.GetRandomRootGroup(GroupType.Large, GroupRole.Admin);
            if (group == null)
            {
                // No group available
                return;
            }

            var request = new ChangeGroupRequest
            {
                AvatarUrl = group.AvatarUrl == ClientConstants.GroupAvatarUrlDefault ? ClientConstants.GroupAvatarUrlAlternate : ClientConstants.GroupAvatarUrlDefault,
                AllowTemporaryChildGroups = !group.AllowTemporaryChildGroups,
                ForcePushToTalk = !group.ForcePushToTalk,
                GroupID = group.GroupID,
                MessageOfTheDay = string.Format("MOTD Changed at {0}", DateTime.UtcNow.ToString("G")),
                Title = string.Format("Group Title Changed at {0}", DateTime.UtcNow.ToString("G"))
            };
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("ChangeGroup", svc => svc.ChangeGroupInfo(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case BasicServiceResponseStatus.Successful:
                    LoadTest.Track(TrackedEventType.ChangedGroupInfo);
                    break;
                case BasicServiceResponseStatus.Forbidden:
                    if (client.MemberOfGroup(group.GroupID, group.RootGroupID, GroupRole.Admin))
                    {
                        Logger.Warn("Failed to Change Group Info", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.ChangeGroupInfoError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Change Group Info", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.ChangeGroupInfoError);
                    break;
            }
        }

        private static void ChangeGroupMemberRole(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var group = client.GetRandomRootGroup(null, GroupRole.Admin);
            if (group == null)
            {
                // No group available
                return;
            }

            var memberToChange = group.Members.FirstOrDefault(m => m.UserID != client.CurrentUser.UserID && m.Role < GroupRole.Admin);
            if (memberToChange == null)
            {
                // No member available
                return;
            }

            var oldRole = memberToChange.Role;
            var request = new ChangeGroupUserRoleRequest
            {
                GroupID = group.GroupID,
                UserID = memberToChange.UserID,
                Role = oldRole < GroupRole.Officer ? GroupRole.Officer : GroupRole.Member
            };
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("ChangeUserRole", svc => svc.ChangeGroupUserRole(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case BasicServiceResponseStatus.Successful:
                    LoadTest.Track(TrackedEventType.ChangedGroupMemberRole);
                    break;
                case BasicServiceResponseStatus.NotFound:
                case BasicServiceResponseStatus.Forbidden:
                    // Check that you are a member of the group and so are they
                    if (client.MemberOfGroup(group.GroupID, group.RootGroupID, GroupRole.Admin) && client.MemberOfGroup(memberToChange.UserID, group.GroupID, group.RootGroupID, oldRole))
                    {
                        Logger.Warn("Failed to Change Group Member Role", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.ChangeGroupMemberRoleError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Change Group Member Role", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.ChangeGroupMemberRoleError);
                    break;
            }
        }

        private static void DeleteRootGroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var rootGroup = client.GetRandomRootGroup(null, GroupRole.Owner);
            if (rootGroup == null)
            {
                // No group available
                return;
            }

            var request = new DeleteGroupRequest {GroupID = rootGroup.GroupID};
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("DeleteGroup", svc => svc.DeleteGroup(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case BasicServiceResponseStatus.Successful:
                    LoadTest.Track(rootGroup.GroupType == GroupType.Normal ? TrackedEventType.NormalGroupDeleted : TrackedEventType.ClanDeleted);
                    break;
                case BasicServiceResponseStatus.NotFound:
                case BasicServiceResponseStatus.Forbidden:
                    if (client.MemberOfGroup(rootGroup.GroupID, rootGroup.RootGroupID, GroupRole.Owner))
                    {
                        Logger.Warn("Failed to Delete Group", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(rootGroup.GroupType == GroupType.Normal ? TrackedEventType.NormalGroupDeletedError : TrackedEventType.ClanDeletedError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Delete Group", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(rootGroup.GroupType == GroupType.Normal ? TrackedEventType.NormalGroupDeletedError : TrackedEventType.ClanDeletedError);
                    break;
            }
        }

        private static void DeleteSubgroup(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var group = client.GetRandomRootGroup(GroupType.Large, GroupRole.Admin);
            if (group == null)
            {
                // No clan available
                return;
            }

            var nonLobby = group.Children.FirstOrDefault(g => !g.IsLobby);
            if (nonLobby == null)
            {
                // No non-lobby subgroup available
                return;
            }

            var request = new DeleteGroupRequest {GroupID = nonLobby.GroupID};
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("DeleteSubgroup", svc => svc.DeleteGroup(request));
            client.QueueActionOrEvent(response);

            switch (response.Status)
            {
                case BasicServiceResponseStatus.Successful:
                    LoadTest.Track(TrackedEventType.SubgroupDeleted);
                    break;
                case BasicServiceResponseStatus.NotFound:
                case BasicServiceResponseStatus.Forbidden:
                    if (client.MemberOfGroup(nonLobby.GroupID, nonLobby.RootGroupID, GroupRole.Admin))
                    {
                        Logger.Warn("Failed to Delete Subgroup", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.SubgroupDeletedError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to Delete Subgroup", new {client.CurrentUser.UserID, request, response});
                    LoadTest.Track(TrackedEventType.SubgroupDeletedError);
                    break;
            }
        }

        #endregion

        #region State-Based Behaviors

        private static void RespondToFriendRequest(LoadTests.Client.Behavior.ClientBehavior client, Random random)
        {
            var randomFriends = client.GetRandomFriends().Where(f => f.FriendshipStatus == FriendshipStatus.AwaitingMe).ToArray();
            if (randomFriends.Length == 0)
            {
                // No friends to confirm
                return;
            }

            var willDecline = random.Next(0, 99) < 30;
            var unconfirmed = randomFriends.First();

            if (willDecline)
            {
                const bool blockFutureRequests = false; // TODO: Make this random when the production client supports this.
                var request = new DeclineFriendshipRequest {FriendID = unconfirmed.UserID, BlockFutureRequests = blockFutureRequests};
                client.QueueActionOrEvent(request);
                var response = client.FriendsWebClient.Call("DeclineFriend", svc => svc.DeclineFriendship(request));
                client.QueueActionOrEvent(response);

                switch (response.Status)
                {
                    case DeclineFriendshipStatus.Successful:
                        LoadTest.Track(TrackedEventType.FriendRequestDeclinedByMe);
                        break;
                    case DeclineFriendshipStatus.NotFound:
                        if (client.FriendsWith(unconfirmed.UserID, FriendshipStatus.AwaitingMe))
                        {
                            Logger.Warn("Failed to Decline Friendship", new {client.CurrentUser.UserID, request, response});
                            LoadTest.Track(TrackedEventType.FriendRequestDeclineError);
                        }
                        break;
                    default:
                        Logger.Warn("Failed to Decline Friendship", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.FriendRequestDeclineError);
                        break;
                }
            }
            else
            {
                var request = new ConfirmFriendshipRequest {FriendID = unconfirmed.UserID};
                client.QueueActionOrEvent(request);
                var response = client.FriendsWebClient.Call("ConfirmFriend", svc => svc.ConfirmFriendship(request));
                client.QueueActionOrEvent(response);

                switch (response.Status)
                {
                    case ConfirmFriendshipStatus.Successful:
                        LoadTest.Track(TrackedEventType.FriendRequestAcceptedByMe);
                        break;
                    case ConfirmFriendshipStatus.NotFound:
                        if (client.FriendsWith(unconfirmed.UserID, FriendshipStatus.AwaitingMe))
                        {
                            Logger.Warn("Failed to Confirm Friendship", new {client.CurrentUser.UserID, request, response});
                            LoadTest.Track(TrackedEventType.FriendRequestAcceptedError);
                        }
                        break;
                    default:
                        Logger.Warn("Failed to Confirm Friendship", new {client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.FriendRequestAcceptedError);
                        break;
                }
            }
        }

        #endregion

        #region General

        private static void DoNothing(LoadTests.Client.Behavior.ClientBehavior client)
        {
            // Do nothing!
        }

        private static void Logout(LoadTests.Client.Behavior.ClientBehavior client)
        {
            client.Logout();
        }

        private static void HangUp(LoadTests.Client.Behavior.ClientBehavior client)
        {
            client.VoiceManager.HangUp();
        }

        private static void ChangeConnectionStatus(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var newStatus = (UserConnectionStatus) ((int) (client.CurrentUser.ConnectionStatus + 1)%5) + 1;
            var request = new ChangeStatusRequest {MachineKey = ClientConstants.MachineKey, Status = newStatus, CustomStatusMessage = newStatus.ToString()};
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("ChangeStatus", svc => svc.ChangeStatus(request));
            client.QueueActionOrEvent(response);
        }

        private static void ChangeProfile(LoadTests.Client.Behavior.ClientBehavior client)
        {
            GetProfile(client);

            var useAlternates = client.CurrentUser.AvatarUrl == ClientConstants.UserAvatarUrlDefault;

            var request = new ChangeProfileRequest
            {
                AboutMe = useAlternates ? "I'm awesome" : "I'm Salty",
                AvatarUrl = useAlternates ? ClientConstants.UserAvatarUrlAlternate : ClientConstants.UserAvatarUrlDefault,
                City = useAlternates ? "Huntsville" : "Essex",
                CountryCode = useAlternates ? "US" : "UK",
                Name = useAlternates ? client.CurrentUser.Username : "Sir " + client.CurrentUser.Username,
                State = useAlternates ? "AL" : "Brentwood"
            };
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("ChangeProfile", svc => svc.ChangeProfile(request));
            client.QueueActionOrEvent(response);

            if (response.Status == ChangeProfileStatus.Successful)
            {
                LoadTest.Track(TrackedEventType.ProfileChanged);
                client.CurrentUser.AboutMe = request.AboutMe;
                client.CurrentUser.City = request.City;
                client.CurrentUser.CountryCode = request.CountryCode;
                client.CurrentUser.Name = request.Name;
                client.CurrentUser.State = request.State;
                client.CurrentUser.AvatarUrl = request.AvatarUrl;
            }
            else
            {
                Logger.Warn("Failed to change profile", new {client.CurrentUser.UserID, request, response});
                LoadTest.Track(TrackedEventType.ProfileChangeError);
            }
        }

        private static void GetProfile(LoadTests.Client.Behavior.ClientBehavior client)
        {
            client.QueueActionOrEvent(string.Format("GetProfile for {0}", client.CurrentUser.UserID));
            var response = client.FriendsWebClient.Call("GetProfile", svc => svc.GetUserProfile(client.CurrentUser.UserID));
            client.QueueActionOrEvent(response);

            if (response.Status == BasicServiceResponseStatus.Successful)
            {
                client.CurrentUser.AboutMe = response.AboutMe;
                client.CurrentUser.City = response.City;
                client.CurrentUser.CountryCode = response.CountryCode;
                client.CurrentUser.Name = response.Name;
                client.CurrentUser.State = response.State;
                client.CurrentUser.AvatarUrl = response.AvatarUrl;
            }
            else
            {
                Logger.Warn("Failed to get profile", new {client.CurrentUser.UserID, response});
            }
        }

        private static void ChangeGameStatus(LoadTests.Client.Behavior.ClientBehavior client)
        {
            var newGame = client.CurrentUser.GameID + 1%2;
            var request = new ChangeGameStatusRequest {GameID = newGame, GameState = newGame, IsRunning = newGame > 0, MachineKey = ClientConstants.MachineKey.ToString()};
            client.QueueActionOrEvent(request);
            var response = client.FriendsWebClient.Call("ChangeGameStatus", svc => svc.ChangeGameStatus(request));
            client.QueueActionOrEvent(response);

            if (response.Status != ChangeGameStatusStatus.Successful)
            {
                Logger.Warn("Failed to change game status", new {client.CurrentUser.UserID, request, response});
            }
        }

        #endregion
    }
}
