﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Curse.ClientBehavior.Behavior;
using Curse.Friends.Client;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.LoadTests.Client.Models;
using Curse.LoadTests.Client.ServiceClients;
using Curse.LoadTests.Client.Utilities;
using Curse.LoadTests.Enums;
using Curse.Logging;
using Curse.ServiceAuthentication.Models;
using Curse.SocketInterface;

namespace Curse.LoadTests.Client.Behavior
{
    class ClientBehavior
    {
        public FriendsWebClient FriendsWebClient { get; private set; }

        public VoiceWebClient VoiceWebClient { get; private set; }

        public FriendsNotificationClient NotificationClient { get { return _notificationClient; } }

        public VoiceManager VoiceManager { get; private set; }

        public LoggedInUser CurrentUser { get; private set; }

        public ConcurrentDictionary<int, Friend> Friends { get; private set; }

        public ConcurrentDictionary<Guid, Group> Groups { get; private set; }

        private readonly Random _random;
        private FriendsNotificationClient _notificationClient;

        public ClientBehavior(int userID, string username, int randomSeed)
        {
            CurrentUser = new LoggedInUser
            {
                UserID = userID,
                Username = username
            };

            Friends = new ConcurrentDictionary<int, Friend>();
            Groups = new ConcurrentDictionary<Guid, Group>();

            _random = new Random(randomSeed);
            FriendsWebClient = new FriendsWebClient(userID);
            VoiceWebClient = new VoiceWebClient(userID);
            VoiceManager = new VoiceManager(this, _random);
        }

        public bool IsLoggedIn
        {
            get { return NotificationClient != null && NotificationClient.IsConnected; }
        }

        private bool _hasLoggedIn;

        private CancellationToken _cancellationToken;
        private int _minActionDelayMillis;
        private int _maxActionDelayMillis;

        public void Start(CancellationToken token, int minActionDelayMillis, int maxActionDelayMillis)
        {
            _cancellationToken = token;
            _minActionDelayMillis = minActionDelayMillis;
            _maxActionDelayMillis = maxActionDelayMillis;
            Continue();
        }

        private void Continue()
        {
            if (_cancellationToken.IsCancellationRequested)
            {
                Logout();
                return;
            }

            Task.Delay(_hasLoggedIn ? _random.Next(_minActionDelayMillis, _maxActionDelayMillis) : _random.Next(_minActionDelayMillis, (int) TimeSpan.FromMinutes(5).TotalMilliseconds),
                _cancellationToken)
                .ContinueWith(t =>
                {
                    if (t.Status != TaskStatus.RanToCompletion)
                    {
                        return;
                    }
                    DoSomething();
                    Continue();
                }, _cancellationToken);
        }

        private void DoSomething()
        {
            try
            {
                if (_cancellationToken.IsCancellationRequested)
                {
                    Logout();
                    return;
                }

                Actions.PerformAction(this, _random);

            }
            catch (OperationCanceledException)
            {
                // Do nothing, exit requested
            }
        }

        #region Friends

        public Friend[] GetRandomFriends()
        {
            return Friends.Values.OrderBy(f => _random.Next()).ToArray();
        }

        public Friend[] GetRandomFriends(int minNumber, int maxNumber)
        {
            return Friends.Values.OrderBy(f => _random.Next()).Take(_random.Next(minNumber, maxNumber)).ToArray();
        }

        public bool FriendsWith(int userID, FriendshipStatus? expectedStatus = null)
        {
            Friend friend;
            return Friends.TryGetValue(userID, out friend) && (expectedStatus == null || friend.FriendshipStatus == expectedStatus.Value);
        }

        #endregion

        #region Groups

        public NonFriend GetRandomNonFriend()
        {
            var userToAdd = _random.Next(99) < 10 
                // Potentially Cross-Region Friends
                ? LoadTestUsers.GetRandomUsers(50).FirstOrDefault(u => u.UserID != CurrentUser.UserID && !Friends.ContainsKey(u.UserID)) 
                // Only in-region friends
                : LoadTestUsers.GetRandomUsersInMyRegion(50).FirstOrDefault(u => u.UserID != CurrentUser.UserID && !Friends.ContainsKey(u.UserID));

            if (userToAdd == null)
            {
                return null;
            }

            return new NonFriend
            {
                Username = userToAdd.Username,
                UserID = userToAdd.UserID
            };
        }

        public Group GetRandomRootGroup(GroupType? type = null, GroupRole? minRole = null)
        {
            var groups = type == null ? Groups.Values : Groups.Values.Where(g => g.GroupType == type.Value);
            var group = groups.OrderBy(g => _random.Next()).FirstOrDefault();
            if (group == null)
            {
                // No groups available
                return null;
            }

            if (!group.HasLoadedDetails)
            {
                var request = new GroupDetailsRequest
                {
                    GroupID = group.GroupID,
                    IncludePermissions = false,
                    IncludeRoleNames = false,
                    ShowDeletedChannels = false
                };
                QueueActionOrEvent(request);
                var response = FriendsWebClient.Call("GetGroupDetails", svc => svc.GetGroupDetails(request));
                QueueActionOrEvent(response);

                if (response.Status == BasicServiceResponseStatus.Successful)
                {
                    group.LoadDetails(response.GroupMembership, response.Groups, response.Members);
                }
                else
                {
                    Group temp;
                    Groups.TryRemove(group.GroupID, out temp);
                    return null;
                }
            }

            return minRole == null || group.Members.First(m => m.UserID == CurrentUser.UserID).Role >= minRole ? group : null;
        }

        public Group GetRandomGroup()
        {
            var root = GetRandomRootGroup();
            if (root == null)
            {
                return null;
            }

            var myRole = root.Members.First(m => m.UserID == CurrentUser.UserID).Role;

            var list = new List<Group>(root.Children.Where(g => g.AccessLevel <= myRole)) {root};
            return list.OrderBy(g => _random.Next()).First();
        }

        public Group GetRandomSubgroup()
        {
            var root = GetRandomRootGroup(GroupType.Large);
            if (root == null)
            {
                return null;
            }

            return root.Children.OrderBy(g => _random.Next()).FirstOrDefault();
        }

        public bool MemberOfGroup(Guid groupID, Guid rootGroupID, GroupRole? expectedRole = null)
        {
            return MemberOfGroup(CurrentUser.UserID, groupID, rootGroupID, expectedRole);
        }

        public bool MemberOfGroup(int userID, Guid groupID, Guid rootGroupID, GroupRole? expectedRole = null)
        {
            Group root;
            if (!Groups.TryGetValue(rootGroupID, out root))
            {
                return false;
            }

            var group = groupID == rootGroupID ? root : root.Children.FirstOrDefault(g => g.GroupID == groupID);
            if (group == null)
            {
                return false;
            }

            var member = root.Members.FirstOrDefault(m => m.UserID == userID);
            if (member == null)
            {
                return false;
            }

            return member.Role >= group.AccessLevel && (expectedRole == null || member.Role == expectedRole);
        }

        public bool MemberOfGroup(Guid groupID, GroupRole? expectedRole = null)
        {
            Group group = null;
            Group rootGroup;
            if (!Groups.TryGetValue(groupID, out rootGroup))
            {
                foreach (var root in Groups.Values)
                {
                    group = root.Children.FirstOrDefault(g => g.GroupID == groupID);
                    if (group != null)
                    {
                        rootGroup = root;
                        break;
                    }
                }
            }

            if (rootGroup == null || group == null)
            {
                return false;
            }

            var member = rootGroup.Members.FirstOrDefault(m => m.UserID == CurrentUser.UserID);
            if (member == null)
            {
                return false;
            }

            return member.Role >= group.AccessLevel && (expectedRole == null || member.Role == expectedRole);
        }

        #endregion

        #region Connection

        public bool Login()
        {
            var loginRequest = new LoginRequest {Username = CurrentUser.Username, EncryptedPassword = LoginWebClient.EncryptLocalString("P@ssw0rd")};
            QueueActionOrEvent(loginRequest);
            var loginResponse = LoginWebClient.Instance.Call("Login", client => client.Login(loginRequest));
            QueueActionOrEvent(loginResponse);

            if (loginResponse.Status != AuthenticationStatus.Success)
            {
                Logger.Warn("Failed to Authenticate", new {CurrentUser.UserID, loginRequest, loginResponse});
                LoadTest.Track(TrackedEventType.LoginError);
                return false;
            }

            CurrentUser.AuthToken = loginResponse.Session.Token;
            FriendsWebClient.UpdateToken(loginResponse.Session.Token);
            VoiceWebClient.UpdateToken(loginResponse.Session.Token);

            var registerRequest = new RegisterSelfRequest
            {
                DeviceID = null,
                MachineKey = ClientConstants.MachineKey,
                Platform = DevicePlatform.Windows,
                Status = UserConnectionStatus.Online
            };
            QueueActionOrEvent(registerRequest);
            var registerResponse = FriendsWebClient.Call("RegisterSelf", client => client.RegisterSelf(registerRequest));
            QueueActionOrEvent(registerResponse);

            if (registerResponse.Status != RegisterSelfStatus.Successful)
            {
                Logger.Warn("Failed to Register Self", new { CurrentUser.UserID, registerRequest, registerResponse });
                LoadTest.Track(TrackedEventType.LoginError);
                return false;
            }

            var connectionInfo = new NotificationConnectionInfo
            {
                Ports = registerResponse.NotificationHostPorts,
                UserID = CurrentUser.UserID,
                MachineKey = ClientConstants.MachineKey.ToString(),
                NotificationHosts = registerResponse.NotificationHostList,
                SessionID = registerResponse.SessionID,
                ClientVersion = "6.0.5000.0",
                DesiredStatus = UserConnectionStatus.Online,
                
            };
            _random.Shuffle(connectionInfo.NotificationHosts);

            JoinResponse joinResponse;
            _notificationClient = FriendsNotificationClient.Connect(connectionInfo, out joinResponse);
            QueueActionOrEvent(joinResponse);

            if (NotificationClient != null)
            {
                NotificationClient.Disconnected += NotificationClient_Disconnected;
            }

            if (joinResponse.Status != JoinStatus.Successful)
            {
                Logger.Warn("Failed to Connect to a Notification Server", new {CurrentUser.UserID, joinResponse});
                LoadTest.Track(TrackedEventType.LoginError);
                return false;
            }

            LoadTest.Track(TrackedEventType.Login);
            _hasLoggedIn = true;
            return true;
        }

        public void Logout()
        {
            VoiceManager.HangUp(true);

            var notificationClient = _notificationClient;
            if (notificationClient != null)
            {
                LoadTest.Track(TrackedEventType.Logout);
                QueueActionOrEvent("Logout");
                notificationClient.Disconnect(SocketDisconnectReason.UserInitiated);
            }
        }

        void NotificationClient_Disconnected(object sender, SocketDisconnectEventArgs e)
        {
            QueueActionOrEvent(string.Format("Notification Session Ended: {0}", e.DisconnectReason));
            LoadTest.Track(TrackedEventType.NotificationServerDisconnect);
            var client = (FriendsNotificationClient)sender;
            client.Disconnected -= NotificationClient_Disconnected;
            Interlocked.CompareExchange(ref _notificationClient, null, client);
        }

        #endregion

        public void QueueActionOrEvent(object actionOrEvent)
        {
#if DEBUG
            _actionsandEvents.Enqueue(actionOrEvent);
#endif
        }

#if DEBUG
        private readonly ConcurrentQueue<object> _actionsandEvents = new ConcurrentQueue<object>();

        public bool TryDequeueActionOrEvent(out object actionOrEvent)
        {
            return _actionsandEvents.TryDequeue(out actionOrEvent);
        }
#endif
    }
}
