﻿using System;
using System.Linq;
using System.Threading.Tasks;
using Curse.Friends.Client;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;
using Curse.LoadTests.Client.Models;
using Curse.LoadTests.Enums;
using Curse.Logging;

namespace Curse.LoadTests.Client.Behavior
{
    internal class NotificationReceiver
    {
        public static void Init()
        {
            FriendsNotificationClient.FriendshipChangeNotification += FriendsNotificationClient_FriendshipChangeNotification;
            FriendsNotificationClient.FriendshipRemovedNotification += FriendsNotificationClient_FriendshipRemovedNotification;

            FriendsNotificationClient.InstantMessageNotification += FriendsNotificationClient_InstantMessageNotification;
            FriendsNotificationClient.InstantMessageResponse += FriendsNotificationClient_InstantMessageResponse;

            FriendsNotificationClient.GroupChangeNotification += FriendsNotificationClient_GroupChangeNotification;

            FriendsNotificationClient.GroupMessageNotification += FriendsNotificationClient_GroupMessageNotification;
            FriendsNotificationClient.GroupMessageResponse += FriendsNotificationClient_GroupMessageResponse;

            FriendsNotificationClient.UserChangeNotification += FriendsNotificationClient_UserChangeNotification;

            FriendsNotificationClient.GroupVoiceDeclineNotification += FriendsNotificationClient_GroupVoiceDeclineNotification;
            FriendsNotificationClient.GroupVoiceInvitationNotification += FriendsNotificationClient_GroupVoiceInvitationNotification;

            FriendsNotificationClient.VoiceDeclineNotification += FriendsNotificationClient_VoiceDeclineNotification;
            FriendsNotificationClient.VoiceInvitationNotification += FriendsNotificationClient_VoiceInvitationNotification;

            FriendsNotificationClient.VoiceAcceptNotification += FriendsNotificationClient_VoiceAcceptNotification;
        }

        private static ClientBehavior GetClient(int userID)
        {
            ClientBehavior client;
            if (LoadTest.Instance.Clients.TryGetValue(userID, out client))
            {
                return client;
            }
            return null;
        }

        private static void FriendsNotificationClient_VoiceInvitationNotification(object sender, SocketMessages.EventArgs<VoiceInvitationNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            LoadTest.Track(TrackedEventType.FriendCallReceived);

            Task.Delay(TimeSpan.FromSeconds(10)).ContinueWith(_ =>
            {
                client.VoiceManager.RespondToCall(e.Value.SenderID, e.Value.InviteUrl, null, e.Value.AccessToken);
            });
        }

        private static void FriendsNotificationClient_VoiceAcceptNotification(object sender, SocketMessages.EventArgs<VoiceAcceptNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);
        }

        private static void FriendsNotificationClient_VoiceDeclineNotification(object sender, SocketMessages.EventArgs<VoiceDeclineNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);
        }

        private static void FriendsNotificationClient_GroupVoiceInvitationNotification(object sender, SocketMessages.EventArgs<GroupVoiceInvitationNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            LoadTest.Track(TrackedEventType.GroupCallReceived);

            Task.Delay(TimeSpan.FromSeconds(10)).ContinueWith(_ =>
            {
                client.VoiceManager.RespondToCall(e.Value.RequestorID, e.Value.InviteUrl, e.Value.GroupID, null);
            });
        }

        private static void FriendsNotificationClient_GroupVoiceDeclineNotification(object sender, SocketMessages.EventArgs<GroupVoiceDeclineNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);
        }

        private static void FriendsNotificationClient_UserChangeNotification(object sender, SocketMessages.EventArgs<UserChangeNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            client.CurrentUser.ConnectionStatus = e.Value.User.ConnectionStatus;
            client.CurrentUser.AvatarUrl = e.Value.User.AvatarUrl;
            client.CurrentUser.GameID = e.Value.User.CurrentGameID;
            client.CurrentUser.GameState = e.Value.User.CurrentGameState;
        }

        private static void FriendsNotificationClient_GroupMessageResponse(object sender, SocketMessages.EventArgs<GroupMessageResponse> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            switch (e.Value.Status)
            {
                case DeliveryStatus.Successful:
                    LoadTest.Track(TrackedEventType.GroupMessageSent);
                    break;
                case DeliveryStatus.Forbidden:
                    if (client.MemberOfGroup(e.Value.GroupID))
                    {
                        Logger.Warn("Failed to send Group Message", new { client.CurrentUser.UserID, e.Value });
                        LoadTest.Track(TrackedEventType.GroupMessageError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to send Group Message", new { client.CurrentUser.UserID, e.Value });
                    LoadTest.Track(TrackedEventType.GroupMessageError);
                    break;
            }
        }

        private static void FriendsNotificationClient_GroupMessageNotification(object sender, SocketMessages.EventArgs<GroupMessageNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            LoadTest.Track(TrackedEventType.GroupMessageReceived);
        }

        private static void FriendsNotificationClient_GroupChangeNotification(object sender, SocketMessages.EventArgs<GroupChangeNotification> e)
        {
            var clientID = ((FriendsNotificationClient) sender).ClientID;
            var client = GetClient(clientID);
            client.QueueActionOrEvent(e.Value);

            Group rootGroup;
            switch (e.Value.ChangeType)
            {
                case GroupChangeType.AddUsers:
                    if (client.Groups.TryGetValue(e.Value.Group.RootGroupID, out rootGroup))
                    {
                        rootGroup.AddOrUpdateMembers(e.Value.Members);
                    }
                    break;
                case GroupChangeType.ChangeInfo:
                    break;
                case GroupChangeType.CreateGroup:
                    if (client.Groups.TryGetValue(e.Value.Group.GroupID, out rootGroup))
                    {
                        rootGroup.UpdateFromNotification(e.Value.Group);
                    }
                    else if (e.Value.Group.GroupID == e.Value.Group.RootGroupID)
                    {
                        client.Groups.TryAdd(e.Value.Group.GroupID, new Group(e.Value.Group, e.Value.Members, e.Value.ChildGroups));
                    }
                    break;
                case GroupChangeType.GroupReorganized:
                    // TODO: Implement
                    break;
                case GroupChangeType.PermissionsChanged:
                    // TODO: Implement
                    break;
                case GroupChangeType.RemoveGroup:
                    if (client.Groups.TryGetValue(e.Value.Group.GroupID, out rootGroup))
                    {
                        client.Groups.TryRemove(rootGroup.GroupID, out rootGroup);
                    }
                    else if (client.Groups.TryGetValue(e.Value.Group.RootGroupID, out rootGroup))
                    {
                        rootGroup.RemoveSubgroup(e.Value.Group.GroupID);
                    }
                    break;
                case GroupChangeType.RemoveUsers:
                    if (e.Value.Members.Any(m => m.UserID == clientID))
                    {
                        client.Groups.TryRemove(e.Value.Group.GroupID, out rootGroup);
                    }
                    else if (client.Groups.TryGetValue(e.Value.Group.RootGroupID, out rootGroup))
                    {
                        rootGroup.RemoveMembers(e.Value.Members.Select(m => m.UserID).ToArray());
                    }
                    break;
                case GroupChangeType.RoleNamesChanged:
                    // TODO: Implement
                    break;
                case GroupChangeType.UpdateUsers:
                    if (client.Groups.TryGetValue(e.Value.Group.RootGroupID, out rootGroup))
                    {
                        rootGroup.UpdateMembers(e.Value.Members);
                    }
                    break;
                case GroupChangeType.VoiceSessionEnded:
                    if (client.Groups.TryGetValue(e.Value.Group.RootGroupID, out rootGroup))
                    {
                        if (e.Value.Group.GroupID == rootGroup.GroupID)
                        {
                            rootGroup.UpdateFromNotification(e.Value.Group);
                        }
                        else
                        {
                            rootGroup.AddOrUpdateGroup(e.Value.Group);
                        }
                    }
                    break;
                case GroupChangeType.VoiceSessionStarted:
                    if (client.Groups.TryGetValue(e.Value.Group.RootGroupID, out rootGroup))
                    {
                        if (e.Value.Group.GroupID == rootGroup.GroupID)
                        {
                            rootGroup.UpdateFromNotification(e.Value.Group);
                        }
                        else
                        {
                            rootGroup.AddOrUpdateGroup(e.Value.Group);
                        }
                    }
                    break;
                case GroupChangeType.VoiceSessionUserJoined:
                    if (client.Groups.TryGetValue(e.Value.Group.RootGroupID, out rootGroup))
                    {
                        rootGroup.UpdateMembers(e.Value.Members);
                    }
                    break;
                case GroupChangeType.VoiceSessionUserLeft:
                    if (client.Groups.TryGetValue(e.Value.Group.RootGroupID, out rootGroup))
                    {
                        rootGroup.UpdateMembers(e.Value.Members);
                    }
                    break;
            }
        }

        private static void FriendsNotificationClient_InstantMessageNotification(object sender, SocketMessages.EventArgs<InstantMessageNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            LoadTest.Track(TrackedEventType.FriendMessageReceived);
        }

        private static void FriendsNotificationClient_InstantMessageResponse(object sender, SocketMessages.EventArgs<InstantMessageResponse> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            switch (e.Value.Status)
            {
                case DeliveryStatus.Successful:
                    LoadTest.Track(TrackedEventType.FriendMessageSent);
                    break;
                case DeliveryStatus.Forbidden:
                    if (client.FriendsWith(e.Value.RecipientID, FriendshipStatus.Confirmed))
                    {
                        Logger.Warn("Failed to send Instant Message", new {client.CurrentUser.UserID, e.Value});
                        LoadTest.Track(TrackedEventType.FriendMessageError);
                    }
                    break;
                default:
                    Logger.Warn("Failed to send Instant Message", new { client.CurrentUser.UserID, e.Value });
                    LoadTest.Track(TrackedEventType.FriendMessageError);
                    break;
            }
        }

        private static void FriendsNotificationClient_FriendshipRemovedNotification(object sender, SocketMessages.EventArgs<FriendshipRemovedNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            Friend removed;
            client.Friends.TryRemove(e.Value.FriendID, out removed);

            //LoadTestStatistics.Track(TrackedEventType.RemoveFriendReceived);
        }

        private static void FriendsNotificationClient_FriendshipChangeNotification(object sender, SocketMessages.EventArgs<FriendshipChangeNotification> e)
        {
            var client = GetClient(((FriendsNotificationClient)sender).ClientID);
            client.QueueActionOrEvent(e.Value);

            if (client.Friends.All(f => f.Key != e.Value.Friendship.OtherUserID))
            {
                LoadTest.Track(TrackedEventType.FriendRequestReceived);
            }

            client.Friends.AddOrUpdate(e.Value.Friendship.OtherUserID,
                v => new Friend(e.Value.Friendship),
                (id, f) => f.UpdateFromNotification(e.Value.Friendship));
        }
    }
}
