﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Configuration.CurseVoiceService;
using Curse.Friends.Enums;
using Curse.LoadTests.Enums;
using Curse.Logging;
using Curse.SocketInterface;
using Curse.SocketMessages;
using Curse.Voice.Client;
using Curse.Voice.Contracts;
using Curse.Voice.Enums;

namespace Curse.LoadTests.Client.Behavior
{
    class VoiceManager
    {
        private readonly object _callLock = new object();
        private readonly ClientBehavior _client;
        private readonly Random _random;

        private VoiceClient _voiceClient;
        private DateTime _lastVoiceConnection = DateTime.MinValue;
        private DateTime _lastVoiceEnd = DateTime.MinValue;

        private Dictionary<int, int> _activeCallers = new Dictionary<int, int>();
        private Dictionary<int, string> _pendingCallers = new Dictionary<int, string>();

        public VoiceManager(ClientBehavior client, Random random)
        {
            _client = client;
            _random = random;
        }

        public void CallFriend(int friendID)
        {
            lock (_callLock)
            {
                if (_voiceClient != null || !CallCooldownExpired)
                {
                    return;
                }

                var request = new CallFriendRequest
                {
                    ClientVersion = "6.0.65535.65535",
                    FriendID = friendID,
                    SendInvitation = true
                };
                _client.QueueActionOrEvent(request);
                var response = _client.FriendsWebClient.Call("CallFriend", svc => svc.CallFriend(request));
                _client.QueueActionOrEvent(response);

                switch (response.Status)
                {
                    case VoiceSessionRequestStatus.Successful:
                        JoinSessionResponse joinResponse;
                        GetVoiceSessionResponse getSessionResponse;
                        if (DoConnectToCall(response.InviteUrl, response.AccessToken, null, friendID, out getSessionResponse, out joinResponse))
                        {
                            LoadTest.Track(TrackedEventType.FriendCalled);
                        }
                        else
                        {
                            Logger.Warn("Failed to Call Friend", new { _client.CurrentUser.UserID, request, getSessionResponse, joinResponse });
                            LoadTest.Track(TrackedEventType.FriendCallError);
                        }
                        break;
                    case VoiceSessionRequestStatus.Forbidden:
                        if (_client.FriendsWith(friendID, FriendshipStatus.Confirmed))
                        {
                            LoadTest.Track(TrackedEventType.FriendCallError);
                            Logger.Warn("Failed to Call Friend", new { _client.CurrentUser.UserID, request, response });
                        }
                        break;
                    default:
                        Logger.Warn("Failed to Call Friend", new { _client.CurrentUser.UserID, request, response });
                        LoadTest.Track(TrackedEventType.FriendCallError);
                        break;
                }
            }
        }

        public void CallGroup(Guid groupID)
        {
            lock (_callLock)
            {
                if (_voiceClient != null || !CallCooldownExpired)
                {
                    return;
                }

                var request = new CallGroupRequest
                {
                    ClientVersion = "6.0.65535.65535",
                    GroupID = groupID,
                    SendInvitation = true
                };
                _client.QueueActionOrEvent(request);
                var response = _client.FriendsWebClient.Call("CallGroup", svc => svc.CallGroup(request));
                _client.QueueActionOrEvent(response);

                switch (response.Status)
                {
                    case VoiceSessionRequestStatus.Successful:
                        JoinSessionResponse joinResponse;
                        GetVoiceSessionResponse getSessionResponse;
                        if (DoConnectToCall(response.InviteUrl, response.AccessToken, groupID, null, out getSessionResponse, out joinResponse))
                        {
                            LoadTest.Track(TrackedEventType.GroupCalled);
                        }
                        else
                        {
                            Logger.Warn("Failed to Call group", new {_client.CurrentUser.UserID, request, getSessionResponse, joinResponse});
                            LoadTest.Track(TrackedEventType.GroupCallError);
                        }
                        break;
                    case VoiceSessionRequestStatus.Forbidden:
                        if (_client.MemberOfGroup(groupID))
                        {
                            Logger.Warn("Failed to Call group", new {_client.CurrentUser.UserID, request, response});
                            LoadTest.Track(TrackedEventType.GroupCallError);
                        }
                        break;
                    default:
                        Logger.Warn("Failed to Call group", new {_client.CurrentUser.UserID, request, response});
                        LoadTest.Track(TrackedEventType.GroupCallError);
                        break;
                }
            }
        }

        private bool DoConnectToCall(string inviteCode, long accessToken, Guid? groupID, int? friendID, out GetVoiceSessionResponse getSessionResponse, out JoinSessionResponse joinResponse)
        {
            _client.QueueActionOrEvent(string.Format("GetVoiceSession {0}", inviteCode));
            getSessionResponse = _client.VoiceWebClient.Call("GetVoiceSession", svc => svc.GetVoiceSession(inviteCode, ClientConstants.ClientVersion));
            _client.QueueActionOrEvent(getSessionResponse);

            if (getSessionResponse.Status != GetVoiceSessionStatus.Successful)
            {
                joinResponse = null;
                Logger.Warn("Unable to find session for voice call", new {inviteCode, response = getSessionResponse});
                return false;
            }

            var connectionInfo = new VoiceConnectionInfo(new CodecInfo { Name = "Opus", PacketMilliseconds = 40 })
            {
                GroupID = groupID ?? Guid.Empty,
                UserID = _client.CurrentUser.UserID,
                FriendID = friendID ?? 0,
                AvatarUrl = _client.CurrentUser.AvatarUrl,
                GameID = getSessionResponse.GameID,
                Port = getSessionResponse.Port,
                AccessToken = accessToken,
                AuthToken = _client.CurrentUser.AuthToken,
                DisplayName = _client.CurrentUser.Username,
                HostID = getSessionResponse.HostID,
                HostName = getSessionResponse.HostName,
                IPAddress = IPAddress.Parse(getSessionResponse.IPAddress),
                InstanceID = getSessionResponse.InstanceCode,
                InviteUrl = inviteCode,
                OriginalDisplayName = _client.CurrentUser.Username,
                RegionName = getSessionResponse.RegionName,
                SessionType = friendID == null ? VoiceInstanceType.Group : VoiceInstanceType.Friend,
            };

            _voiceClient = VoiceClient.Connect(new[] { getSessionResponse.Port, 3784, 6100, 9987, 10011 }, connectionInfo, false, ClientConstants.ClientVersion.ToString(), out joinResponse);
            _client.QueueActionOrEvent(joinResponse);

            if (_voiceClient!=null && joinResponse.Status==JoinSessionStatus.Successful)
            {
                _lastVoiceConnection = DateTime.UtcNow;
                Subscribe(_voiceClient);
                _voiceClient.GetUserList();
                return true;
            }

            return false;
        }

        /// <summary>
        /// There can be a race where a friend call ends and right away you try to call it back = SessionNotFound on join request, but successful everywhere else
        /// </summary>
        private bool CallCooldownExpired
        {
            get { return DateTime.UtcNow - _lastVoiceEnd > TimeSpan.FromSeconds(10); }
        }

        #region Voice Host Events

        private void Subscribe(VoiceClient client)
        {
            client.DisconnectedSession += VoiceClient_DisconnectedSession;
            client.UserListUpdated += VoiceClient_UserListUpdated;
            client.UserJoined += VoiceClient_UserJoined;
            client.UserDisconnected += VoiceClient_UserDisconnected;
            client.UserLeft += VoiceClient_UserLeft;
            client.AddPendingUsers += VoiceClient_AddPendingUsers;
            client.RemovePendingUsers += VoiceClient_RemovePendingUsers;
        }

        private void Unsubscribe(VoiceClient client)
        {
            client.DisconnectedSession -= VoiceClient_DisconnectedSession;
            client.UserListUpdated -= VoiceClient_UserListUpdated;
            client.UserJoined -= VoiceClient_UserJoined;
            client.UserDisconnected -= VoiceClient_UserDisconnected;
            client.UserLeft -= VoiceClient_UserLeft;
            client.AddPendingUsers -= VoiceClient_AddPendingUsers;
            client.RemovePendingUsers -= VoiceClient_RemovePendingUsers;
        }

        private void VoiceClient_DisconnectedSession(object sender, SocketDisconnectEventArgs e)
        {
            lock (_callLock)
            {
                _client.QueueActionOrEvent(string.Format("Voice Session Ended: {0}", e.DisconnectReason));
                LoadTest.Track(TrackedEventType.VoiceServerDisconnect);

                var voiceClient = (VoiceClient)sender;
                voiceClient.DisconnectedSession -= VoiceClient_DisconnectedSession;

                if (_voiceClient == voiceClient)
                {
                    Unsubscribe(_voiceClient);
                    _voiceClient = null;
                    _lastVoiceEnd = DateTime.UtcNow;
                }
            }
        }

        private void VoiceClient_UserLeft(object sender, EventArgs<UserLeftNotification> e)
        {
            lock (_callLock)
            {
                if (sender == _voiceClient)
                {
                    _activeCallers.Remove(e.Value.UserID);
                }
            }
        }

        private void VoiceClient_UserDisconnected(object sender, EventArgs<UserDisconnectNotification> e)
        {
            lock (_callLock)
            {
                if (sender == _voiceClient)
                {
                    _activeCallers.Remove(e.Value.AffectedUserID);
                }
            }
        }

        private void VoiceClient_UserJoined(object sender, EventArgs<UserJoinedNotification> e)
        {
            lock (_callLock)
            {
                if (sender == _voiceClient)
                {
                    if (!_activeCallers.ContainsKey(e.Value.User.ClientID))
                    {
                        _activeCallers.Add(e.Value.User.ClientID, e.Value.User.UserID ?? 0);
                    }
                }
            }
        }

        private void VoiceClient_UserListUpdated(object sender, EventArgs<GetUsersResponse> e)
        {
            lock (_callLock)
            {
                if (sender == _voiceClient)
                {
                    _activeCallers = e.Value.Users == null ? new Dictionary<int, int>() : e.Value.Users.ToDictionary(u => u.ClientID, u => u.UserID ?? 0);
                    _pendingCallers = e.Value.PendingUsers == null ? new Dictionary<int, string>() : e.Value.PendingUsers.ToDictionary(pu => pu.UserID, pu => pu.DisplayName);
                }
            }
        }

        private void VoiceClient_RemovePendingUsers(object sender, EventArgs<RemovePendingUserNotification> e)
        {
            lock (_callLock)
            {
                if (sender == _voiceClient)
                {
                    _pendingCallers.Remove(e.Value.UserID);
                }
            }
        }

        private void VoiceClient_AddPendingUsers(object sender, EventArgs<AddPendingUsersNotification> e)
        {
            lock (_callLock)
            {
                if (sender == _voiceClient)
                {
                    foreach (var pendingCaller in e.Value.Users)
                    {
                        if (!_pendingCallers.ContainsKey(pendingCaller.UserID))
                        {
                            _pendingCallers.Add(pendingCaller.UserID, pendingCaller.DisplayName);
                        }
                    }
                }
            }
        }

        #endregion

        public void HangUp(bool force=false)
        {
            lock (_callLock)
            {
                if (_voiceClient == null)
                {
                    return;
                }

                var timeSinceJoined = DateTime.UtcNow - _lastVoiceConnection;
                if (force || timeSinceJoined > TimeSpan.FromMinutes(5) || (IsAloneInCall() && timeSinceJoined > TimeSpan.FromMinutes(1)))
                {
                    // Hang up only if you've been in the call long enough or it is a forced disconnect
                    _client.QueueActionOrEvent("HangUp");
                    _voiceClient.Disconnect(SocketDisconnectReason.UserInitiated);
                }
            }
        }

        private bool IsAloneInCall()
        {
            return _activeCallers.Count == 1 && _pendingCallers.Count == 0;
        }

        public void RespondToCall(int senderID, string inviteUrl, Guid? groupID, long? accessToken)
        {
            lock (_callLock)
            {
                if (_voiceClient == null && CallCooldownExpired)
                {
                    // Not currently in a call
                    if (_random.Next(0, 99) < 50)
                    {
                        DeclineOrIgnoreCall(senderID, inviteUrl, groupID);
                    }
                    else
                    {
                        AcceptCall(senderID, inviteUrl, groupID, accessToken);
                    }
                }
                else
                {
                    DeclineOrIgnoreCall(senderID, inviteUrl, groupID);
                }
            }
        }

        private void DeclineOrIgnoreCall(int senderID, string inviteUrl, Guid? groupID)
        {
            if (_random.Next(0, 99) < 25)
            {
                // Ignore call
                return;
            }

            var request = new RespondToCallRequest
            {
                Accepted = false,
                InviteUrl = inviteUrl,
                FriendID = groupID.HasValue ? null : (int?)senderID,
                GroupID = groupID
            };
            _client.QueueActionOrEvent(request);
            var response = _client.FriendsWebClient.Call("RespondToCall", svc => svc.RespondToCall(request));
            _client.QueueActionOrEvent(response);

            if (groupID.HasValue)
            {
                switch (response.Status)
                {
                    case RespondToCallStatus.Successful:
                        LoadTest.Track(TrackedEventType.GroupCallDeclined);
                        break;
                    case RespondToCallStatus.Forbidden:
                    case RespondToCallStatus.NotFound:
                        if (_client.MemberOfGroup(groupID.Value))
                        {
                            Logger.Warn("Failed to decline group call", new { _client.CurrentUser.UserID, request, response });
                            LoadTest.Track(TrackedEventType.GroupCallError);
                        }
                        break;
                    default:
                        Logger.Warn("Failed to decline group call", new { _client.CurrentUser.UserID, request, response });
                        LoadTest.Track(TrackedEventType.GroupCallError);
                        break;
                }
            }
            else
            {
                switch (response.Status)
                {
                    case RespondToCallStatus.Successful:
                        LoadTest.Track(TrackedEventType.FriendCallDeclinedByMe);
                        break;
                    case RespondToCallStatus.Forbidden:
                    case RespondToCallStatus.NotFound:
                        if (_client.FriendsWith(senderID, FriendshipStatus.Confirmed))
                        {
                            Logger.Warn("Failed to decline friend call", new { _client.CurrentUser.UserID, request, response });
                            LoadTest.Track(TrackedEventType.FriendCallError);
                        }
                        break;
                    default:
                        Logger.Warn("Failed to decline friend call", new { _client.CurrentUser.UserID, request, response });
                        LoadTest.Track(TrackedEventType.FriendCallError);
                        break;
                }
            }
        }

        private void AcceptCall(int senderID, string inviteUrl, Guid? groupID, long? accessToken)
        {
            var respondToCallRequest = new RespondToCallRequest
            {
                Accepted = true,
                InviteUrl = inviteUrl,
                FriendID = groupID.HasValue ? null : (int?)senderID,
                GroupID = groupID
            };
            _client.QueueActionOrEvent(respondToCallRequest);
            var respondToCallResponse = _client.FriendsWebClient.Call("RespondToCall", svc => svc.RespondToCall(respondToCallRequest));
            _client.QueueActionOrEvent(respondToCallResponse);

            if (groupID.HasValue)
            {
                switch (respondToCallResponse.Status)
                {
                    case RespondToCallStatus.Successful:
                        var callGroupRequest = new CallGroupRequest
                        {
                            ClientVersion = ClientConstants.ClientVersion.ToString(),
                            GroupID = groupID.Value,
                            SendInvitation = false
                        };
                        _client.QueueActionOrEvent(callGroupRequest);
                        var callGroupResponse = _client.FriendsWebClient.Call("CallGroup", svc => svc.CallGroup(callGroupRequest));
                        _client.QueueActionOrEvent(callGroupResponse);

                        switch (callGroupResponse.Status)
                        {
                            case VoiceSessionRequestStatus.Successful:
                                JoinSessionResponse joinResponse;
                                GetVoiceSessionResponse getSessionResponse;
                                if (DoConnectToCall(inviteUrl, callGroupResponse.AccessToken, groupID.Value, null, out getSessionResponse, out joinResponse))
                                {
                                    LoadTest.Track(TrackedEventType.GroupCallAcceptedByMe);
                                }
                                else
                                {
                                    Logger.Warn("Failed to accept group call", new { _client.CurrentUser.UserID, respondToCallRequest, getSessionResponse, joinResponse });
                                    LoadTest.Track(TrackedEventType.GroupCallError);
                                }
                                break;
                            case VoiceSessionRequestStatus.Forbidden:
                                if (_client.MemberOfGroup(groupID.Value))
                                {
                                    Logger.Warn("Failed to accept group call", new {_client.CurrentUser.UserID, respondToCallRequest, respondToCallResponse, callGroupRequest, callGroupResponse});
                                    LoadTest.Track(TrackedEventType.GroupCallError);
                                }
                                break;
                            default:
                                Logger.Warn("Failed to accept group call", new {_client.CurrentUser.UserID, respondToCallRequest, respondToCallResponse, callGroupRequest, callGroupResponse});
                                LoadTest.Track(TrackedEventType.GroupCallError);
                                break;
                        }
                        break;
                    case RespondToCallStatus.Forbidden:
                    case RespondToCallStatus.NotFound:
                        if (_client.MemberOfGroup(groupID.Value))
                        {
                            Logger.Warn("Failed to accept group call", new { _client.CurrentUser.UserID, respondToCallRequest, respondToCallResponse });
                            LoadTest.Track(TrackedEventType.GroupCallError);
                        }
                        break;
                    default:
                        Logger.Warn("Failed to accept group call", new { _client.CurrentUser.UserID, request = respondToCallRequest, respondToCallResponse });
                        LoadTest.Track(TrackedEventType.GroupCallError);
                        break;
                }
            }
            else if (accessToken.HasValue)
            {
                switch (respondToCallResponse.Status)
                {
                    case RespondToCallStatus.Successful:
                        JoinSessionResponse joinResponse;
                        GetVoiceSessionResponse getSessionResponse;
                        if (DoConnectToCall(inviteUrl, accessToken.Value, null, senderID, out getSessionResponse, out joinResponse))
                        {
                            LoadTest.Track(TrackedEventType.FriendCallAcceptedByMe);
                        }
                        else
                        {
                            Logger.Warn("Failed to accept friend call", new { _client.CurrentUser.UserID, request = respondToCallRequest, getSessionResponse, joinResponse });
                            LoadTest.Track(TrackedEventType.FriendCallError);
                        }
                        LoadTest.Track(TrackedEventType.FriendCallAcceptedByMe);
                        break;
                    case RespondToCallStatus.Forbidden:
                    case RespondToCallStatus.NotFound:
                        if (_client.FriendsWith(senderID, FriendshipStatus.Confirmed))
                        {
                            Logger.Warn("Failed to accept friend call", new { _client.CurrentUser.UserID, request = respondToCallRequest, response = respondToCallResponse });
                            LoadTest.Track(TrackedEventType.FriendCallError);
                        }
                        break;
                    default:
                        Logger.Warn("Failed to accept friend call", new { _client.CurrentUser.UserID, request = respondToCallRequest, response = respondToCallResponse });
                        LoadTest.Track(TrackedEventType.FriendCallError);
                        break;
                }
            }
        }
    }
}
