﻿using System;
using System.Diagnostics;
using System.Net;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.ServiceModel.Description;
using System.Threading.Tasks;
using Curse.LoadTests.Contracts;
using Curse.LoadTests.Contracts.Response;
using Curse.LoadTests.Enums;
using Curse.Logging;

namespace Curse.LoadTests.Client
{
    [ServiceBehavior(InstanceContextMode = InstanceContextMode.Single)]
    public class ClientServiceHost : IClientService
    {
        public static readonly ClientServiceHost Instance = new ClientServiceHost();
        private ServiceHost _serviceHost;

        public void Initialize()
        {
            _serviceHost = new ServiceHost(Instance);
#if CONFIG_CONSOLEAPP
            var endpointAddress = string.Format("http://{0}/client-service/{1}", Environment.MachineName.ToLower(), Process.GetCurrentProcess().Id);
#elif CONFIG_WINDOWSSERVICE
            var endpointAddress = string.Format("http://{0}/client-service", Environment.MachineName.ToLower());
#else
            var endpointAddress = string.Format("http://{0}/client-service-debug",Environment.MachineName.ToLower());
#endif
            _serviceHost.AddServiceEndpoint(new ServiceEndpoint(ContractDescription.GetContract(typeof (IClientService)), new BasicHttpBinding(),
                new EndpointAddress(endpointAddress)));
            _serviceHost.Open();
        }

        public void Terminate()
        {
            var host = _serviceHost;
            if (host != null)
            {
                host.Close();
            }
        }

        private ClientServiceHost()
        {
            Logger.Info("Starting Client Host service");
        }

        public GetClientStatusResponse GetClientStatus()
        {
            return new GetClientStatusResponse {Status = LoadTest.Instance.Status};
        }

        public StartResponse Start(int startInd, int endInd)
        {
            try
            {
                var status = LoadTest.Instance.Status;
                if (status == ClientStatus.Running || status == ClientStatus.Starting || status == ClientStatus.Stopping)
                {
                    return new StartResponse {Status = StartStatus.AlreadyRunning};
                }

                Task.Factory.StartNew(() =>
                {
                    Logger.Info("Starting Client Behavior", new {startInd, endInd});
                    LoadTest.Instance.Start(startInd, endInd, DateTime.UtcNow.Ticks.GetHashCode());
                });

                return new StartResponse {Status = StartStatus.Successful};
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Error while starting Client Behavior");
                return new StartResponse {Status = StartStatus.Error};
            }
        }

        public void Stop()
        {
            Logger.Info("Client Behavior Is Stopping");

            Task.Factory.StartNew(() =>
            {
                LoadTest.Instance.Stop();
            });
        }
    }
}
