﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.IO;
using System.Threading;
using System.Threading.Tasks;
using Curse.LoadTests.Client.Configuration;
using Curse.LoadTests.Client.Utilities;
using Curse.LoadTests.Enums;
using Curse.LoadTests.Statistics;
using Curse.Logging;

namespace Curse.LoadTests.Client
{
    class LoadTest
    {
        public static readonly LoadTest Instance = new LoadTest();

        private LoadTest()
        {
            ChangeStatus(ClientStatus.Ready);
        }

        private CancellationTokenSource _cts;

        public readonly ConcurrentDictionary<int, Behavior.ClientBehavior> Clients = new ConcurrentDictionary<int, Behavior.ClientBehavior>();

        public event EventHandler StatusChanged;

        public ClientStatus Status { get; private set; }

        private void ChangeStatus(ClientStatus status)
        {
            try
            {
                if (Status == status)
                {
                    return;
                }

                Status = status;
                if (StatusChanged != null)
                {
                    StatusChanged(this, new EventArgs());
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to change client status to " + status);
            }
        }

        public void Start(int startIndex, int endIndex, int randomSeed)
        {
            if (Status != ClientStatus.Ready && Status != ClientStatus.Error)
            {
                Logger.Error("Client cannot be started from its current state: " + Status);
                throw new InvalidOperationException();
            }

            var users = LoadTestUsers.GetUsers(startIndex, endIndex - startIndex + 1);
            Start(users, randomSeed);
        }

        public void Start(LoadTestUsers.UserInfo[] users, int randomSeed)
        {
            ChangeStatus(ClientStatus.Starting);

            _cts = new CancellationTokenSource();

            var baseRandom = new Random(randomSeed);
            foreach (var userinfo in users)
            {
                var client = new Behavior.ClientBehavior(userinfo.UserID, userinfo.Username, baseRandom.Next());
                Clients.TryAdd(userinfo.UserID, client);
            }

            ClientHostStatisticsManager.Initialize(ClientConfiguration.Instance.LocalRegionID, Environment.MachineName, ClientConfiguration.Instance.StatisticsIntervalMinutes,
                ClientConfiguration.Instance.MonitorServiceUrl, ClientConfiguration.Instance.CentralServiceApiKey);

            foreach (var kvp in Clients)
            {
                kvp.Value.Start(_cts.Token, ClientConstants.MinActionSleepMillis, ClientConstants.MaxActionSleepMillis);
            }

            ChangeStatus(ClientStatus.Running);
        }

        public void Stop()
        {
            
            ChangeStatus(ClientStatus.Stopping);

            if (_cts != null)
            {
                _cts.Cancel();
                _cts = null;
            }

            Parallel.ForEach(Clients, client => client.Value.Logout());
            ClientHostStatisticsManager.Stop();
            Clients.Clear();

            ChangeStatus(ClientStatus.Ready);
        }

        #region Statistics

        public static void Track(TrackedEventType type)
        {
#if DEBUG
            Interlocked.Increment(ref Stats[type].Value);
#else
            try
            {
                ClientHostStatisticsManager.Current.Get("Tester").TrackStatForEvent(type);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to track event", new {type});
            }
#endif
        }


#if DEBUG
        private class SimpleStat
        {
            public int Value;
        }

        private static readonly Dictionary<TrackedEventType, SimpleStat> Stats = new Dictionary<TrackedEventType, SimpleStat>();

        static LoadTest()
        {
            foreach(TrackedEventType eventType in  Enum.GetValues(typeof(TrackedEventType)))
            {
                Stats[eventType]=new SimpleStat();
            }
        }

        public static void WriteResults()
        {
            if (!Directory.Exists("results"))
            {
                Directory.CreateDirectory("results");
            }

            using (var stream = new StreamWriter("results/countresults.txt"))
            {
                foreach (var kvp in Stats)
                {
                    stream.WriteLine("{0}\t{1}", kvp.Key, kvp.Value.Value);
                }
            }
        }
#endif

        #endregion
    }
}
