﻿using System;
using System.Collections.Concurrent;
using System.Linq;
using Curse.Friends.Client.FriendsService;
using Curse.Friends.Enums;
using Curse.Friends.NotificationContracts;

namespace Curse.LoadTests.Client.Models
{
    class Group
    {
        public Guid GroupID { get; set; }

        public Guid ParentGroupID { get; set; }

        public Guid RootGroupID { get; set; }

        public GroupType GroupType { get; set; }

        public string Title { get; set; }

        public string MessageOfTheDay { get; set; }

        public string AvatarUrl { get; set; }

        public bool ForcePushToTalk { get; set; }

        public bool AllowTemporaryChildGroups { get; set; }

        public bool IsLobby { get; set; }

        public GroupRole AccessLevel { get; set; }

        private readonly ConcurrentDictionary<int, GroupMemberNotification> _members;
        private readonly ConcurrentDictionary<Guid, Group> _children;

        public Group(GroupNotification notification, GroupMemberNotification[] members, GroupNotification[] children)
            : this(notification)
        {
            foreach (var member in members)
            {
                _members.TryAdd(member.UserID, member);
            }

            foreach (var child in children)
            {
                _children.TryAdd(child.GroupID, new Group(child));
            }
        }

        public Group(GroupNotification notification)
        {
            GroupID = notification.GroupID;
            ParentGroupID = notification.ParentGroupID;
            RootGroupID = notification.RootGroupID;
            GroupType = notification.GroupType;
            IsLobby = notification.IsDefaultChannel;

            if (GroupID == RootGroupID)
            {
                _children = new ConcurrentDictionary<Guid, Group>();
                _members = new ConcurrentDictionary<int, GroupMemberNotification>();
            }

            UpdateFromNotification(notification);

        }

        public Group(GroupMembership membership)
        {
            GroupID = membership.GroupID;
            ParentGroupID = membership.ParentGroupID;
            RootGroupID = membership.RootGroupID;
            GroupType = membership.GroupType;

            if (GroupID == RootGroupID)
            {
                _children = new ConcurrentDictionary<Guid, Group>();
                _members = new ConcurrentDictionary<int, GroupMemberNotification>();
            }

            UpdateFromGroupMembership(membership);

        }

        public Group UpdateFromNotification(GroupNotification notification)
        {
            Title = notification.GroupTitle;
            AvatarUrl = notification.GroupAvatar;
            AccessLevel = notification.GroupAccessLevel;
            return this;
        }


        public Group UpdateFromGroupMembership(GroupMembership membership)
        {
            Title = membership.GroupTitle;
            AvatarUrl = membership.GroupAvatarUrl;
            AccessLevel = membership.GroupRole;
            return this;
        }

        public bool HasLoadedDetails { get; private set; }

        public void LoadDetails(GroupMembership membership, GroupNotification[] groups, GroupMemberNotification[] members)
        {
            if (GroupID != RootGroupID)
            {
                throw new InvalidOperationException();
            }

            // Remove members not in the details set and add or update those who are
            RemoveMembers(_members.Values.Select(m => m.UserID).Where(id => members.All(em => em.UserID != id)).ToArray());
            AddOrUpdateMembers(members);

            // Remove children not in the details set and add or update those who are
            foreach (var groupID in _children.Keys.Where(id => groups.All(g => g.GroupID != id && g.GroupID != GroupID)))
            {
                RemoveSubgroup(groupID);
            }
            foreach (var group in groups.Where(g => g.GroupID != GroupID))
            {
                AddOrUpdateGroup(group);
            }

            HasLoadedDetails = true;
        }

        public GroupMemberNotification[] Members { get { return GroupID == RootGroupID ? _members.Values.ToArray() : new GroupMemberNotification[0]; } }

        public bool IsMember(int userID)
        {
            return GroupID == RootGroupID && _members.ContainsKey(userID);
        }

        public void AddOrUpdateMembers(GroupMemberNotification[] members)
        {
            if (GroupID != RootGroupID)
            {
                return;
            }

            foreach (var member in members)
            {
                _members.AddOrUpdate(member.UserID,
                    id => member,
                    (id, m) =>
                    {
                        m.CurrentGroupID = member.CurrentGroupID;
                        m.Role = member.Role;
                        return m;
                    });
            }
        }

        public void UpdateMembers(GroupMemberNotification[] members)
        {
            if (GroupID != RootGroupID)
            {
                return;
            }

            foreach (var updated in members)
            {
                GroupMemberNotification member;
                if (!_members.TryGetValue(updated.UserID, out member))
                {
                    // Not found in member list
                    continue;
                }

                member.CurrentGroupID = updated.CurrentGroupID;
                member.Role = updated.Role;
            }
        }

        public void RemoveMembers(int[] ids)
        {
            if (GroupID != RootGroupID)
            {
                return;
            }

            foreach (var id in ids)
            {
                GroupMemberNotification temp;
                _members.TryRemove(id, out temp);
            }
        }

        public Group[] Children { get { return GroupID == RootGroupID ? _children.Values.ToArray() : new Group[0]; } }

        public void AddOrUpdateGroup(GroupNotification group)
        {
            if (GroupID != RootGroupID)
            {
                return;
            }

            _children.AddOrUpdate(group.GroupID,
                id => new Group(group),
                (id, g) => g.UpdateFromNotification(group));
        }

        public void AddOrUpdateSubgroup(GroupMembership membership)
        {
            if (GroupID != RootGroupID)
            {
                return;
            }

            _children.AddOrUpdate(membership.GroupID,
                id => new Group(membership),
                (id, g) => g.UpdateFromGroupMembership(membership));
        }

        public void RemoveSubgroup(Guid id)
        {
            if (GroupID != RootGroupID)
            {
                return;
            }

            Group temp;
            _children.TryRemove(id, out temp);
        }





    }
}
