﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Aerospike.Client;
using Curse.Aerospike;
using Curse.Friends.Data;
using Curse.Friends.Enums;

namespace Curse.LoadTests.Client.Utilities
{
    public class LoadTestData
    {
        private static readonly HashSet<int> RegionIDs;
        static LoadTestData()
        {
            RegionIDs = new HashSet<int>(AerospikeConfiguration.Configurations.Select(p => p.RegionIdentifier).ToArray().Distinct());
        }

        public static CancellationToken CancellationToken { get; set; }

        private static bool IsCancelled
        {
            get { return CancellationToken != CancellationToken.None && CancellationToken.IsCancellationRequested; }
        }


        public static void WipeTable<T>() where T : BaseTable<T>, new()
        {
            foreach (var regionID in RegionIDs)
            {
                if (IsCancelled)
                {
                    return;
                }
                WipeTable<T>(regionID);
            }
        }

        static void WipeTable<T>(int regionID) where T : BaseTable<T>, new()
        {
            if (IsCancelled)
            {
                return;
            }

            Console.WriteLine("Wiping " + typeof(T).Name + " in region " + regionID);

            try
            {
                BaseTable<T>.BatchOperate(regionID, 100, list =>
                {
                    if (IsCancelled)
                    {
                        throw new AerospikeException.ScanTerminated();
                    }

                    Console.WriteLine("Processing " + list.Count() + " items...");
                    foreach (var model in list)
                    {
                        model.DeleteRemote(regionID);
                    }
                });
            }
            catch (AerospikeException ex)
            {
                if (!(ex.InnerException is AerospikeException.ScanTerminated))
                {
                    throw;
                }
            }

            Console.WriteLine("Done!");
        }

        public static void ResetUserStatus()
        {
            foreach (var regionID in RegionIDs)
            {
                if (IsCancelled)
                {
                    return;
                }

                Console.WriteLine("Resetting user status in region " + regionID);
                var total = 0;
                User.BatchOperate(regionID, 1000, users =>
                {
                    Console.WriteLine("Processing " + users.Count() + " more items...");

                    foreach (var user in users)
                    {
                        if (user.ConnectionStatus != UserConnectionStatus.Offline)
                        {
                            user.ConnectionStatus = UserConnectionStatus.Offline;
                            user.Update(p => p.ConnectionStatus);
                        }
                    }

                    total += users.Count();
                    Console.WriteLine("Finished " + total + "");
                });
            }
        }

        public static void ResetUserStatus(IEnumerable<int> userIDs)
        {
            Console.WriteLine("Resetting user status...");
            foreach (var userID in userIDs)
            {
                if (IsCancelled)
                {
                    return;
                }

                // Get this user's region
                var region = UserRegion.GetLocal(userID);
                if (region == null)
                {
                    Console.WriteLine("Unable to find region for user: " + userID);
                    continue;
                }

                var user = region.GetUser();
                if (user.ConnectionStatus != UserConnectionStatus.Offline)
                {
                    user.ConnectionStatus = UserConnectionStatus.Offline;
                    user.Update(p => p.ConnectionStatus);
                }
            }
        }

        /// <summary>
        /// Resets the affected user's friend count to its proper value
        /// </summary>
        /// <param name="userIDs"></param>
        public static void ResetUserFriendCount(IEnumerable<int> userIDs)
        {
            Console.WriteLine("Resetting user friend count...");
            foreach (var userID in userIDs)
            {
                if (IsCancelled)
                {
                    return;
                }

                // Get this user's region
                var region = UserRegion.GetLocal(userID);
                if (region == null)
                {
                    Console.WriteLine("Unable to find region for user: " + userID);
                    continue;
                }

                var userFriends = Friendship.GetAllConfirmed(region.RegionID, userID);
                var user = region.GetUser();
                if (user.FriendCount != userFriends.Length)
                {
                    user.FriendCount = userFriends.Length;
                    user.Update(p => p.FriendCount);
                }

            }
        }

        /// <summary>
        /// Updates the connection status of every affected user's friends to be offline
        /// </summary>
        /// <param name="userIDs"></param>
        public static void ResetFriendStatus(IEnumerable<int> userIDs)
        {
            Console.WriteLine("Resetting friend status...");
            foreach (var userID in userIDs)
            {
                if (IsCancelled)
                {
                    return;
                }

                // Get this user's region
                var region = UserRegion.GetLocal(userID);
                if (region == null)
                {
                    Console.WriteLine("Unable to find region for user: " + userID);
                    continue;
                }

                var friends = Friendship.GetAll(region.RegionID, p => p.UserID, userID);
                foreach (var friend in friends)
                {
                    if (friend.OtherUserConnectionStatus != UserConnectionStatus.Offline)
                    {
                        friend.OtherUserConnectionStatus = UserConnectionStatus.Offline;
                        friend.Update(p => p.OtherUserConnectionStatus);
                    }
                }
            }
        }

        /// <summary>
        /// Updates all of the affected user ID endpoints to be disconnected
        /// </summary>
        /// <param name="userIDs"></param>
        public static void ResetClientEndpoints(IEnumerable<int> userIDs)
        {
            Console.WriteLine("Resetting endpoints...");
            foreach (var userID in userIDs)
            {
                if (IsCancelled)
                {
                    return;
                }

                // Get this user's region
                var endpoint = ClientEndpoint.GetLocal(userID);
                if (endpoint == null)
                {
                    continue;
                }

                if (endpoint.IsConnected)
                {
                    endpoint.IsConnected = false;
                    endpoint.Update(p => p.IsConnected);
                }
            }
        }
    }
}
