﻿using System;
using System.Collections.Generic;
using System.Data.SQLite;
using System.IO;
using Curse.LoadTests.Contracts;

namespace Curse.LoadTests.Client.Utilities
{
    class LoadTestUsers
    {
        internal static UserInfo[] GetUsers(int startIndex, int numUsers)
        {
            var userInfos = new List<UserInfo>();

            using (var conn = GetReadonlyConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM User WHERE regionID=@regionID AND rowIndex BETWEEN @startIndex AND @endIndex";
                    command.Parameters.AddWithValue("regionID", GetRegionFromHostname(Environment.MachineName));
                    command.Parameters.AddWithValue("startIndex", startIndex);
                    command.Parameters.AddWithValue("endIndex", startIndex + numUsers - 1);

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            userInfos.Add(new UserInfo(reader));
                        }
                    }
                }
            }

            return userInfos.ToArray();
        }

        internal static UserInfo[] GetRandomUsersInMyRegion(int numUsers)
        {
            var userInfos = new List<UserInfo>();

            using (var conn = GetReadonlyConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM User WHERE regionID=@regionID AND rowIndex>=(ABS(RANDOM())%(SELECT MAX(rowIndex) FROM User WHERE regionID=@regionID)) LIMIT @numUsers";
                    command.Parameters.AddWithValue("regionID", GetRegionFromHostname(Environment.MachineName));
                    command.Parameters.AddWithValue("numUsers", numUsers);

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            userInfos.Add(new UserInfo(reader));
                        }
                    }
                }
            }

            return userInfos.ToArray();
        }

        internal static UserInfo[] GetRandomUsers(int numUsers)
        {
            var userInfos = new List<UserInfo>();

            using (var conn = GetReadonlyConnection())
            {
                using (var command = conn.CreateCommand())
                {
                    command.CommandText = "SELECT * FROM User WHERE ROWID>=(ABS(RANDOM())%(SELECT MAX(ROWID) FROM User)) LIMIT @numUsers";
                    command.Parameters.AddWithValue("numUsers", numUsers);

                    using (var reader = command.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            userInfos.Add(new UserInfo(reader));
                        }
                    }
                }
            }

            return userInfos.ToArray();
        }

        private static SQLiteConnection GetReadonlyConnection()
        {
            var conn = new SQLiteConnection(string.Format("Data Source={0};Read Only=True", Path.Combine(AppDomain.CurrentDomain.BaseDirectory, LoadTestConstants.UsersFileName)));
            conn.Open();
            return conn;
        }

        internal class UserInfo
        {
            public UserInfo(SQLiteDataReader reader)
            {
                UserID = reader.GetInt32(reader.GetOrdinal("userID"));
                Username = reader.GetString(reader.GetOrdinal("username"));
                RegionID = reader.GetInt32(reader.GetOrdinal("regionID"));
            }

            public int RegionID { get; set; }

            public int UserID { get; set; }

            public string Username { get; set; }

            public override string ToString()
            {
                return string.Format("UserID = {0}, Username = {1}, RegionID = {2}", UserID, Username, RegionID);
            }
        }

        public static int GetRegionFromHostname(string host)
        {
            if (host.ToLowerInvariant().Contains("dub"))
            {
                return 2;
            }
            if (host.ToLowerInvariant().Contains("sin"))
            {
                return 3;
            }
            // Default to US-East
            return 1;
        }
    }
}
