﻿using System;
using System.IO;
using System.Xml.Serialization;
using Curse.Logging;

namespace Curse.LoadTests.Coordinator.Configuration
{
    public enum ConfigurationMode
    {
        Debug,
        ConsoleApp,
        WindowsService
    }

    [XmlType("LoadTest")]
    public class TestCoordinatorConfiguration
    {
        public static ConfigurationMode Mode { get; private set; }

        private static readonly TestCoordinatorConfiguration _instance = null;

        public static TestCoordinatorConfiguration Instance
        {
            get { return _instance; }
        }

        static TestCoordinatorConfiguration()
        {

#if DEBUG
            Mode = ConfigurationMode.Debug;
#elif CONFIG_CONSOLEAPP
            Mode = ConfigurationMode.ConsoleApp;
#elif CONFIG_WINDOWSSERVICE
            Mode = ConfigurationMode.WindowsService;
#endif
            // Try to load the config, based on the compilation mode

            Logger.Info("Current configuration is: " + Mode);

            var configPath = FindConfigPath();

            Logger.Info("Loading service configuration from: " + configPath);

            using (var reader = new StreamReader(configPath))
            {
                var dcs = new XmlSerializer(typeof(TestCoordinatorConfiguration));
                _instance = (TestCoordinatorConfiguration)dcs.Deserialize(reader);
            }

        }

        private static string FindConfigPath()
        {
            string configPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "Configuration", "TestCoordinator." + Mode.ToString() + ".config");

            if (!File.Exists(configPath))
            {
                configPath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory, "bin", "Configuration", "TestCoordinator." + Mode.ToString() + ".config");
            }

            if (!File.Exists(configPath))
            {
                throw new Exception("Configuration could not be loaded from '" + configPath + "'");
            }

            return configPath;
        }

        public void Save()
        {
            var configPath = FindConfigPath();

            using (var writer = new StreamWriter(configPath))
            {
                var dcs = new XmlSerializer(typeof(TestCoordinatorConfiguration));
                dcs.Serialize(writer, this);
            }
        }

        [XmlArray]
        [XmlArrayItem("Host")]
        public string[] ClientHosts { get; set; }

        [XmlElement]
        public string ClientHostSourcePath { get; set; }

        [XmlElement]
        public string ClientSourcePath { get; set; }

        [XmlElement]
        public int TestDurationMinutes { get; set; }

        [XmlElement]
        public int ClientsPerHost { get; set; }

        [XmlElement]
        public int UsersPerClient { get; set; }

    }
}
