﻿using System;
using System.Collections.Generic;
using System.Configuration;
using System.Data.SqlClient;
using System.Diagnostics;
using System.IO;
using System.Net.Http;
using System.Reflection;
using System.Threading;
using System.Threading.Tasks;
using Curse.Friends.Configuration;
using Curse.Friends.Data;
using Curse.Friends.Data.Models;
using Curse.LoadTests.Client.Utilities;
using Curse.LoadTests.Coordinator.Clients;
using Curse.LoadTests.Coordinator.Configuration;
using Curse.MSBuild.Deployment;
using Timer = System.Threading.Timer;

namespace Curse.LoadTests.Coordinator
{
    public class Program
    {
        private static readonly int MinutesToRun = TestCoordinatorConfiguration.Instance.TestDurationMinutes;
        private static int _percentComplete;
        
        static void Main(string[] args)
        {

            var cts = new CancellationTokenSource();

            Console.CancelKeyPress += (o, e) =>
            {
                Console.WriteLine();
                Console.WriteLine(@"Canceling the load test at the next opportunity...");
                cts.Cancel();
                e.Cancel = true;
            };

            StorageConfiguration.Initialize("CurseLoadTestCoordinator", ConfigurationRegion.USEast.Key);

            LoadTestUser.EnsureUsers(GetNumUsersPerRegion(), cts.Token);

            // Clear aerospike sets
            LoadTestData.CancellationToken = cts.Token;

            LoadTestData.WipeTable<Group>();
            LoadTestData.WipeTable<GroupMemberList>();
            LoadTestData.WipeTable<GroupMembership>();
            LoadTestData.WipeTable<UserClientEndpointMap>();
            LoadTestData.WipeTable<Friendship>();
            //LoadTestData.WipeTable<Conversation>();
            //LoadTestData.WipeTable<GroupEventLog>();

            // Fix aerospike records
            LoadTestData.ResetUserFriendCount(LoadTestUser.UserIDs);
            LoadTestData.ResetUserStatus(LoadTestUser.UserIDs);          
            LoadTestData.ResetFriendStatus(LoadTestUser.UserIDs);
            LoadTestData.ResetClientEndpoints(LoadTestUser.UserIDs);

            PreloadWebSite("http://auth.curse.opt/NetworkService.asmx");
            PreloadWebSite("http://clientservice-v6.curse.opt/ClientLoginService.svc");
            PreloadWebSite("http://friends-service.cursevoice.opt/FriendsService.svc");
            PreloadWebSite("http://voice-service.cursevoice.opt/VoiceService.svc");
            PreloadWebSite("http://loadtestmonitor-service.cursevoice.opt/LoadTestMonitor.svc");

#if CONFIG_CONSOLEAPP && !CONFIG_NODEPLOY
            try
            {
                DeployClientApp();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);

                if (Debugger.IsAttached)
                {
                    Console.WriteLine("Press enter to exit");
                    Console.ReadLine();
                }

                Environment.Exit(1);
            }
#elif CONFIG_WINDOWSSERVICE && !CONFIG_NODEPLOY
            try
            {
                DeployWindowsService();
            }
            catch (Exception ex)
            {
                Console.WriteLine(ex.Message);

                if(Debugger.IsAttached)
                {
                    Console.WriteLine("Press enter to exit");
                    Console.ReadLine();
                }

                return;
            }
#endif

            var clients = CreateClients(cts.Token);

            var startTime = DateTime.UtcNow;
            StartClientBehavior(clients, cts.Token);

            Console.WriteLine(@"Load Test started.");
            if (Debugger.IsAttached)
            {
                // ReSharper disable once MethodSupportsCancellation
                Task.Factory.StartNew(() =>
                {
                    Console.WriteLine(@"Press enter to force terminate.");
                    Console.ReadLine();
                    cts.Cancel();
                });
            }

            using (var timer = new Timer(TimerCallback, cts, TimeSpan.Zero, TimeSpan.FromMinutes(MinutesToRun/(double) 100)))
            {
                cts.Token.WaitHandle.WaitOne();
                timer.Change(Timeout.Infinite, Timeout.Infinite);
                StopClientBehavior(clients);
            }

            Console.WriteLine("Load test ended.");
#if !DEBUG

            Console.WriteLine("Adding Load Test to the DB.");
            //add the start and endtime to loadtest table for reporting purpose
            try
            {
                using (var conn = new SqlConnection(ConfigurationManager.ConnectionStrings["LoadTestTable"].ConnectionString))
                {
                    conn.Open();
                    using (var sqlCmd = conn.CreateCommand())
                    {
                        sqlCmd.CommandText = "INSERT INTO [LoadTestHistory] ([StartDate], [EndDate]) values (@StartDate, @EndDate)";
                        sqlCmd.Parameters.AddWithValue("@StartDate", startTime);
                        sqlCmd.Parameters.AddWithValue("@EndDate", DateTime.UtcNow.AddMinutes(1));
                        sqlCmd.ExecuteNonQuery();
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("failed writing to database: {0}", ex.Message);
            }
#endif
        }

        private static void TimerCallback(object state)
        {
            Console.Title = string.Format("Running Load Test, {0}% complete.", _percentComplete++);
            if (_percentComplete > 100)
            {
                ((CancellationTokenSource) state).Cancel();
            }
        }

        private static int GetRegionFromHostname(string host)
        {
            if (host.ToLowerInvariant().Contains("dub"))
            {
                return 2;
            }
            if (host.ToLowerInvariant().Contains("sin"))
            {
                return 3;
            }
            // Default to US-East
            return 1;
        }

        private static void DeployClientApp()
        {

            var currentDirectory = new DirectoryInfo(Assembly.GetCallingAssembly().Location);
            var loadTestingDirectory = currentDirectory.Parent.Parent.Parent.Parent;
            var clientBinDirectory = new DirectoryInfo(Path.Combine(loadTestingDirectory.FullName, "Curse.LoadTests.Client", "bin", TestCoordinatorConfiguration.Mode.ToString()));

            if (!clientBinDirectory.Exists)
            {
                throw new Exception("Client bin directory does not exist!");
            }

            var deployTask = new ConsoleAppDeploy
            {
                ApplicationName = "CurseLoadTestClient",
                ApplicationSourcePath = clientBinDirectory.FullName,
                ApplicationVersion = "1.0.0.0",
                DriveLetter = "c",
                ServerNames = TestCoordinatorConfiguration.Instance.ClientHosts,
                VersioningDisabled = true,
                UsePowershellForRemoteCommands = true
            };

            if (!deployTask.Execute())
            {
                throw new Exception("Failed to deploy client app!");
            }
        }

        private static void DeployWindowsService()
        {

            var currentDirectory = new DirectoryInfo(Assembly.GetCallingAssembly().Location);
            var loadTestingDirectory = currentDirectory.Parent.Parent.Parent.Parent;
            var clientBinDirectory = new DirectoryInfo(Path.Combine(loadTestingDirectory.FullName, "Curse.LoadTests.Client", "bin", TestCoordinatorConfiguration.Mode.ToString()));

            if (!clientBinDirectory.Exists)
            {
                throw new Exception("Client bin directory does not exist!");
            }

            var deployTask = new AuthenticatedWindowsServiceDeploy
            {
                ApplicationName = "LoadTestClientService",
                ApplicationSourcePath = clientBinDirectory.FullName,
                ApplicationVersion = "1.0.0.0",
                ServerNames = TestCoordinatorConfiguration.Instance.ClientHosts,
                VersioningDisabled = true,
                AutoCommit = true,
                ConfirmCommitToSecondaries = false,
                ServerFilteringPrompt = false,
                ApplicationExecutableFileName = "Curse.LoadTests.Client.exe",
                UsePowershellForRemoteCommands = true
            };

            if (!deployTask.Execute())
            {
                throw new Exception("Failed to deploy client app!");
            }
        }

        private static BaseClient[] CreateClients(CancellationToken token)
        {
            var clients = new List<BaseClient>();
            var indexByRegion = new Dictionary<int, int>();

            foreach (var host in TestCoordinatorConfiguration.Instance.ClientHosts)
            {
                for (int i = 0; i < TestCoordinatorConfiguration.Instance.ClientsPerHost; i++)
                {
                    if (token.IsCancellationRequested)
                    {
                        break;
                    }
                    try
                    {
                        var region = GetRegionFromHostname(host);
                        if (!indexByRegion.ContainsKey(region))
                        {
                            indexByRegion[region] = 0;
                        }
                        var stInd = indexByRegion[region];
                        var endInd = stInd + TestCoordinatorConfiguration.Instance.UsersPerClient - 1;

#if DEBUG
                        var client = new DebugClient(host, stInd, endInd);
#elif CONFIG_WINDOWSSERVICE
                        var client = new WindowsServiceClient(host, stInd, endInd);
#elif CONFIG_CONSOLEAPP
                        var client = ConsoleClient.Create(host, stInd, endInd);
#endif
                        clients.Add(client);

                        indexByRegion[region] += TestCoordinatorConfiguration.Instance.UsersPerClient;
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(@"Failed to create client on {0}. Exception: {1}", host, ex.Message);
                    }
                }
            }
            return clients.ToArray();
        }

        private static void StartClientBehavior(BaseClient[] clients, CancellationToken token)
        {
            if (token.IsCancellationRequested)
            {
                return;
            }

            Parallel.ForEach(clients, client =>
            {
                if (token.IsCancellationRequested)
                {
                    return;
                }

                try
                {
                    client.Start();
                }
                catch (Exception ex)
                {
                    Console.WriteLine(@"Failed to start client {0}. Exception: {1}", client, ex.Message);
                }
            });
        }

        private static void StopClientBehavior(BaseClient[] clients)
        {
            Parallel.ForEach(clients, clientHost =>
            {
                try
                {
                    clientHost.Stop();
                }
                catch (Exception ex)
                {
                    Console.WriteLine(@"Failed to stop client at {0}. Exception: {1}", clientHost, ex.Message);
                }
            });
        }

        private static Dictionary<int, int> GetNumUsersPerRegion()
        {
            var numUsersPerRegion = new Dictionary<int, int>();
            foreach (var host in TestCoordinatorConfiguration.Instance.ClientHosts)
            {
                var regionID = GetRegionFromHostname(host);

                if (!numUsersPerRegion.ContainsKey(regionID))
                {
                    numUsersPerRegion[regionID] = 0;
                }

                numUsersPerRegion[regionID] += (TestCoordinatorConfiguration.Instance.UsersPerClient * TestCoordinatorConfiguration.Instance.ClientsPerHost);
            }
            return numUsersPerRegion;
        }

        private static void PreloadWebSite(string url)
        {
            try
            {
                using (var client = new HttpClient())
                {
                    var result = client.GetAsync(url).Result;
                    if (!result.IsSuccessStatusCode)
                    {
                        Console.WriteLine("Site {0} returned non-successful status code {1}", url, result.StatusCode);
                    }
                }
            }
            catch (Exception ex)
            {
                Console.WriteLine("Exception occurred while loading {0}. {1}", url, ex.Message);
            }
        }
    }
}