﻿using System;
using System.Collections.Concurrent;
using System.Linq;
using Curse.LoadTests.Statistics.Models;

namespace Curse.LoadTests.Statistics
{
    public class ClientHostStatTracker
    {
        private readonly ConcurrentDictionary<string, ClientStatTracker> _statTrackersByProfile = new ConcurrentDictionary<string, ClientStatTracker>();
        private readonly int _regionID;
        private readonly string _machineName;
        private readonly int _statisticsIntervalMinutes;

        public ClientStatTracker Get(string profileName)
        {   
            ClientStatTracker tracker;
            if (!_statTrackersByProfile.TryGetValue(profileName, out tracker))
            {
                tracker = new ClientStatTracker(profileName, _regionID);
                _statTrackersByProfile[profileName] = tracker;
            }
            return tracker;
        }

        private readonly DateTime _timeStamp;

        public ClientHostStatTracker(int regionID, string machineName, int statisticsIntervalMinutes)
        {
            _statisticsIntervalMinutes = statisticsIntervalMinutes;
            _regionID = regionID;
            _machineName = machineName;

            _timeStamp = GetCurrentInterval();
        }

        public bool IsExpired()
        {
            return !_timeStamp.Equals(GetCurrentInterval());
        }

        private DateTime GetCurrentInterval()
        {
            var now = DateTime.UtcNow;
            return new DateTime(now.Year, now.Month, now.Day, now.Hour, now.Minute - now.Minute%_statisticsIntervalMinutes, 0, DateTimeKind.Utc);
        }

        public ClientHostStats ToClientHostStats()
        {
            return new ClientHostStats
            {
                MachineName = _machineName,
                RegionID = _regionID,
                BeginTime = _timeStamp,
                EndTime = _timeStamp.Add(TimeSpan.FromMinutes(_statisticsIntervalMinutes)),
                ClientStatsByBehavior = _statTrackersByProfile.ToDictionary(v=>v.Key, v=>v.Value.GetSnapshot())
            };
        }
    }
}
