﻿using System;
using System.Collections.Concurrent;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.Threading;
using System.Threading.Tasks;
using Curse.LoadTests.Statistics.LoadTestMonitor;
using Curse.Logging;

namespace Curse.LoadTests.Statistics
{
    public class ClientHostStatisticsManager
    {
        private static int _regionID;
        private static string _machineName;
        private static int _statisticsIntervalMinutes = 15;
        private static string _monitorServiceUrl;
        private static string _monitorServiceApiKey;
        private static readonly object LockObject = new object();

        private static ClientHostStatTracker _current;
        public static ClientHostStatTracker Current { get { return _current; } }

        private static readonly BlockingCollection<ClientHostStatTracker> StatsToReport = new BlockingCollection<ClientHostStatTracker>(new ConcurrentQueue<ClientHostStatTracker>());

        private static CancellationTokenSource _cts;

        public static void Initialize(int regionID, string machineName, int statisticsIntervalMinutes, string monitorServiceUrl, string monitorServiceApiKey)
        {
            if (_cts != null && !_cts.IsCancellationRequested)
            {
                Logger.Warn("Statistics manager was reinitialized without ever being stopped! Ignoring.");
                return;
            }

            _cts = new CancellationTokenSource();

            _regionID = regionID;
            _machineName = machineName;
            _statisticsIntervalMinutes = statisticsIntervalMinutes;
            _monitorServiceUrl = monitorServiceUrl;
            _monitorServiceApiKey = monitorServiceApiKey;
            _current = new ClientHostStatTracker(regionID, machineName, statisticsIntervalMinutes);

            Logger.Info("Initialzed stats manager. Stats will be uploaded to: " + _monitorServiceUrl);

            Task.Factory.StartNew(() => ProcessStats(_cts.Token), _cts.Token, TaskCreationOptions.LongRunning, TaskScheduler.Default);
            Task.Factory.StartNew(() => ReportStats(_cts.Token), _cts.Token, TaskCreationOptions.LongRunning, TaskScheduler.Default);
        }

        public static void Stop()
        {

            Logger.Info("Stopping client host statistics manager...");

            try
            {
                var current = Current;
                if (current != null)
                {
                    UploadStats(current);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to submit stats");
            }            
        }

        private static void ProcessStats(CancellationToken token)
        {
            while (!token.IsCancellationRequested)
            {
                lock (LockObject)
                {
                    try
                    {
                        token.WaitHandle.WaitOne(TimeSpan.FromSeconds(5));
                    }
                    catch (OperationCanceledException)
                    {
                        break;
                    }

                    if (!Current.IsExpired())
                    {
                        continue;
                    }

                    StatsToReport.Add(_current, token);

                    _current = new ClientHostStatTracker(_regionID, _machineName, _statisticsIntervalMinutes);
                }
            }
        }

        private static void ReportStats(CancellationToken token)
        {
            while (!token.IsCancellationRequested)
            {
                ClientHostStatTracker tracker = null;
                try
                {
                    tracker = StatsToReport.Take(token);
                    UploadStats(tracker);                    
                }
                catch (OperationCanceledException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    if (tracker != null)
                    {
                        StatsToReport.Add(tracker);
                    }
                    Logger.Warn("[ClientHostStatisticsManager] : failed to upload periodic stats", ex);
                }
            }

            try
            {
                var current = Current;
                if (current != null)
                {
                    UploadStats(current);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to submit stats");
            }            
        }

        private static void UploadStats(ClientHostStatTracker tracker)
        {            
            using (var client = new LoadTestMonitorClient(GetMonitorBinding(), new EndpointAddress(_monitorServiceUrl)))
            {
                client.UploadClientHostStatistics(_monitorServiceApiKey, tracker.ToClientHostStats());
            }
        }

        private static Binding GetMonitorBinding()
        {
            var binding = new BasicHttpBinding()
            {
                OpenTimeout = TimeSpan.FromSeconds(5),
                SendTimeout = TimeSpan.FromSeconds(90),
                ReceiveTimeout = TimeSpan.FromSeconds(10),

            };

            return binding;
        }
    }
}
