﻿using System;
using System.Configuration;

namespace Curse.HealthMonitor
{
    public enum FailureReason
    {
        Configuration = 1,
        Timeout = 2,
        Error = 3,
        Unknown = 4
    }

    public class HealthCheckStatus
    {
        
        private static DateTime? _lastNotification = null;
        private static DateTime? _failureTimestamp = null;
        private static TimeSpan _notificationThrottle = TimeSpan.FromMinutes(5);
        private static int _failureCount = 0;
        private static int _timeoutCount = 0;

        /// <summary>
        /// How many consecutive timeouts must occur before considering the site to be unavailable
        /// </summary>
        private static readonly int TimeoutIncidentThreshold = 6;

        /// <summary>
        /// How many consecutive failures must occur before sending an email alert
        /// </summary>
        private static readonly int FailureAlertThreshold = 10;

        public bool IsSuccessful { get; set; }
        public TimeSpan ResponseTime { get; set; }
        public long ContentLength { get; set; }
        public string ResponseContent { get; set; }
        public FailureReason FailureReason { get; set; }
        public string FailureMessage { get; set; }
        public string Host { get; set; }

        static HealthCheckStatus()
        {
            try
            {
                TimeoutIncidentThreshold = int.Parse(ConfigurationManager.AppSettings["TimeoutIncidentThreshold"]);
                Logger.Log("Timeout incident threshold has been update to " + TimeoutIncidentThreshold, ELogLevel.Info);
            }
            catch
            {
                Logger.Log("The default timeout incident threshold of " + TimeoutIncidentThreshold + " will be used", ELogLevel.Info);
            }

            try
            {
                FailureAlertThreshold = int.Parse(ConfigurationManager.AppSettings["FailureAlertThreshold"]);
                Logger.Log("Failure alert threshold has been update to " + FailureAlertThreshold, ELogLevel.Info);
            }
            catch
            {
                Logger.Log("The default failure alert threshold of " + FailureAlertThreshold + " will be used", ELogLevel.Info);
            }
        }
        

        public HealthCheckStatus(string host, FailureReason failureReason, string failureMessage = null, TimeSpan? responseTime = null)
        {
            Host = host;

            if (failureReason == FailureReason.Timeout)
            {
                ++_timeoutCount;

                if (_timeoutCount < TimeoutIncidentThreshold)
                {
                    IsSuccessful = true;
                    ResponseContent = "Timeout #" + _timeoutCount.ToString();
                    ContentLength = ResponseContent.Length;
                }
            }
            else
            {
                IsSuccessful = false;    
            }

            if (responseTime.HasValue)
            {
                ResponseTime = responseTime.Value;
            }

            FailureReason = failureReason;
            FailureMessage = failureMessage;
            if (!_failureTimestamp.HasValue)
            {
                _failureTimestamp = DateTime.UtcNow;
            }
            ++_failureCount;

            SendAlert();
        }

        public HealthCheckStatus(string host, TimeSpan responseTime, string responseContent)
        {
            Host = host;
            IsSuccessful = true;
            ResponseTime = responseTime;
            ContentLength = responseContent.Length;
            ResponseContent = responseContent;

            _failureCount = 0;
            _timeoutCount = 0;

            SendSuccess();
        }

        private void SendSuccess()
        {
            
            if (!_failureTimestamp.HasValue || !_lastNotification.HasValue)
            {
                return;
            }
            

            TimeSpan timeToResolve = DateTime.UtcNow - _failureTimestamp.Value;

            
            string subject = "[Resolved] " + Host + " on " + Environment.MachineName + " at " + DateTime.Now.ToString("h:mm:ss tt");
            string body = "The health check failure for <b>" + Host + "</b> has been resolved on server <b>" + Environment.MachineName + "</b>. Time to resolve: " + timeToResolve.TotalMinutes.ToString("n0") + " minutes.";
            SimpleMailUtility.SendMessage(subject, body);

            _lastNotification = null;
            _failureTimestamp = null;
        }

        private void SendAlert()
        {           
            if (_failureCount < FailureAlertThreshold)
            {
                return;                
            }

            var lastNotified = _lastNotification.HasValue ? (TimeSpan?)(DateTime.UtcNow - _lastNotification.Value) : null;


            if (lastNotified != null && lastNotified.Value < _notificationThrottle)
            {
                return;
            }

            _lastNotification = DateTime.UtcNow;
         
            var subject = "[" + FailureReason + "] " + Host + " on " + Environment.MachineName + " at " + DateTime.Now.ToString("h:mm:ss tt");
            var body = "<p>The health check for <b>" + Host + "</b> has failed on <b>" + Environment.MachineName + "</b> due to a <b>" + FailureReason.ToString() + "</b>: "
                          + FailureMessage + "</p>"
                          + "<p>This email alert was triggered, because this failure condition occurred <b>" + _failureCount.ToString() + " times</b> in a row.</p>";

            if (lastNotified != null)
            {
                body = body + "<p>You were last notified about this event <b>" + lastNotified.Value.TotalMinutes.ToString("###,##0") + " minutes</b> ago. We will not notify you again for <b>" + _notificationThrottle.TotalMinutes.ToString("###,##0") + " minutes</b></p>";
            }
            else
            {
                body = body + "<p>This is your first notification of this event. We will not notify you again for <b>" + _notificationThrottle.TotalMinutes.ToString("###,##0") + " minutes.</b></p>";
            }
            
            SimpleMailUtility.SendMessage(subject, body);           
        }
    }
}