﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Microsoft.Web.Administration;
using System.Net;
using System.Configuration;
using System.IO;
using System.Net.Mail;

namespace Curse.HealthMonitor
{
    public class HealthMonitorManager
    {
        private static readonly HealthMonitorManager _instance = new HealthMonitorManager();

        public static HealthMonitorManager Instance
        {
            get
            {
                return _instance;
            }
        }

        private string _hostToMonitor = null;
        private string _configIssues = null;
        public string HostToMonitor { get { return _hostToMonitor; } }
        private bool _hostIsNotProvisioned = false;


        public HealthMonitorManager()
        {
            try
            {

                Logger.Log("Health monitor starting...", ELogLevel.Info);

                var emailAddresses = ConfigurationManager.AppSettings["EmailAddresses"].Split(';');
                var emailAddressList = new List<MailAddress>();
                foreach (var a in emailAddresses)
                {
                    try
                    {
                        var mailAddress = new MailAddress(a);
                        emailAddressList.Add(mailAddress);
                    }
                    catch (Exception ex)
                    {
                        _configIssues = "Invalid email address: " + a;
                        break;
                    }

                }

                if (emailAddressList.Count == 0)
                {
                    emailAddressList.Add(new MailAddress("health-check-alerts@curse.com"));
                }

                SimpleMailUtility.MailAddresses = emailAddressList.ToArray();

                var host = ConfigurationManager.AppSettings["host"];
                if (!string.IsNullOrEmpty(host))
                {
                    _hostToMonitor = host;
                    return;
                }

                var iisManager = new ServerManager();
                var sites = iisManager.Sites.Where(p => !p.Bindings.Any(b => b.Host.Equals("health-check.curse.us", StringComparison.InvariantCultureIgnoreCase) && b.Host.IndexOf("cursecdn.com", StringComparison.InvariantCultureIgnoreCase) == -1)).ToArray();

                if (!sites.Any())
                {
                    SimpleMailUtility.SendMessage("[Warning] Health Monitor Configuration Issue on " + Environment.MachineName + " at " + DateTime.Now.ToString("h:mm:ss tt"), "The health was not able to determine the site URL to monitor.");
                    _hostIsNotProvisioned = true;
                    return;
                }

                // Get the first IIS Site with a Started state;
                var firstNonHealthCheckSite = sites
                    .Where(p => !p.Name.Equals("HealthMonitor"))
                    .OrderBy(s => s.Id)
                    .FirstOrDefault(p => p.State == ObjectState.Started);

                if (firstNonHealthCheckSite == null)
                {
                    SimpleMailUtility.SendMessage("[Fatal] Health Monitor Failure on " + Environment.MachineName + " at " + DateTime.Now.ToString("h:mm:ss tt"),
                        "The health monitor was unable to identify a web site to monitor.");
                    Logger.Log("Health monitor was not able to identify a web site to monitor.", ELogLevel.Error);
                    return;
                }

                if (firstNonHealthCheckSite.Bindings.Any(p => p.Host.StartsWith("www.", StringComparison.InvariantCultureIgnoreCase)))
                {
                    _hostToMonitor = firstNonHealthCheckSite.Bindings.First(p => p.Host.StartsWith("www.", StringComparison.InvariantCultureIgnoreCase)).Host;
                }
                else
                {
                    var bestBinding = firstNonHealthCheckSite.Bindings.FirstOrDefault(p => !string.IsNullOrEmpty(p.Host))
                                      ?? firstNonHealthCheckSite.Bindings.First();

                    _hostToMonitor = bestBinding.Host;
                }


                Logger.Log("Health monitor started. It is monitoring: " + _hostToMonitor, ELogLevel.Info);
                SimpleMailUtility.SendMessage("[Info] Health Monitor Started on " + Environment.MachineName + " at " + DateTime.Now.ToString("h:mm:ss tt"), "The health monitor has started on " + Environment.MachineName + ", and will monitor the web application at: " + _hostToMonitor);

            }
            catch (Exception ex)
            {
                Logger.Log("Health monitor failed to start: " + ex.Message + ", Stack: " + ex.StackTrace, ELogLevel.Error);
                SimpleMailUtility.SendMessage("[Error] Health Monitor Failed to Start on " + Environment.MachineName + " at " + DateTime.Now.ToString("h:mm:ss tt"), "The health monitor has failed to start on " + Environment.MachineName + ". Error Message: " + ex.Message + ". Stack Trace: " + ex.StackTrace);
            }

        }

        public HealthCheckStatus CheckSite()
        {
            HttpContext.Current.Response.TrySkipIisCustomErrors = true;

            if (_hostIsNotProvisioned)
            {
                return new HealthCheckStatus("None", TimeSpan.FromMilliseconds(100), "This node is not yet provisioned with a web application.");
            }

            if (_hostToMonitor == null)
            {
                return new HealthCheckStatus(_hostToMonitor, FailureReason.Configuration, "Health monitor was unable to auto-detect the host to monitor, and one is not defined in the configuration. To correct this, ensure that IIS has a web site configured with at least one non-empty binding.");
            }

            var startTime = DateTime.UtcNow;

            try
            {
                string result = GetWebPage("http://127.0.0.1", 1);

                if (result.Length == 0)
                {
                    throw new WebException("Empty response");
                }

                return new HealthCheckStatus(_hostToMonitor, DateTime.UtcNow - startTime, result);
            }
            catch (WebException ex)
            {
                if (ex.Response != null)
                {
                    using (var reader = new StreamReader(ex.Response.GetResponseStream()))
                    {
                        var result = reader.ReadToEnd();
                        if (!string.IsNullOrEmpty(result) && result.Contains("cobalt-app-offline-htm"))
                        {
                            return new HealthCheckStatus(_hostToMonitor, DateTime.UtcNow - startTime, result);
                        }
                    }
                }

                Logger.Log("Healthcheck failed due to a web exception:" + ex.Message, ELogLevel.Warning);

                if (ex.Status == WebExceptionStatus.Timeout)
                {
                    return new HealthCheckStatus(_hostToMonitor, FailureReason.Timeout, "The request to " + _hostToMonitor + " timed out.", DateTime.UtcNow - startTime);
                }
                else
                {
                    return new HealthCheckStatus(_hostToMonitor, FailureReason.Error, ex.Message);
                }
            }
            catch (Exception ex)
            {
                Logger.Log("Healthcheck failed due to a unhandled exception:" + ex.Message, ELogLevel.Warning);
                return new HealthCheckStatus(_hostToMonitor, FailureReason.Unknown, "Check to " + _hostToMonitor + " failed: " + ex.Message);
            }
        }

        private string GetWebPage(string url, int count)
        {
            if (count > 5)
            {
                throw new HttpException("Number of redirects exceeded configured amount.");
            }
            HttpWebRequest webRequest = (HttpWebRequest)WebRequest.Create(url);
            webRequest.UserAgent = "googlebot (Curse Health Monitor)";
            webRequest.Proxy = null;
            webRequest.Host = _hostToMonitor;
            webRequest.Timeout = 2000; // 1.5 Seconds
            webRequest.AllowAutoRedirect = false;

            DateTime startTime = DateTime.UtcNow;
            HttpWebResponse response = (HttpWebResponse)webRequest.GetResponse();
            DateTime endTime = DateTime.UtcNow;
            string result = null;
            using (var reader = new StreamReader(response.GetResponseStream()))
            {
                result = reader.ReadToEnd();
            }

            if (response.Headers.AllKeys.Contains("Location")) // This is a redirect
            {
                string location = response.Headers["Location"];
                int charsToSkip = 0;
                string protocol = "http://";

                if (location.StartsWith("http://"))
                {
                    charsToSkip = 7;
                }
                else if (location.StartsWith("https://"))
                {
                    protocol = "https://";
                    charsToSkip = 8;
                }

                if (charsToSkip > 0)
                {
                    location = location.Substring(charsToSkip);
                    string[] parts = location.Split('/');
                    location = protocol + "127.0.0.1/" + string.Join("/", parts.Skip(1));
                }
                else
                {
                    location = protocol + "127.0.0.1" + location;
                }
                return GetWebPage(location, ++count);
            }

            return result;
        }

    }
}