﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Xml.Linq;
using Microsoft.Build.Framework;
using System.Configuration;
using Microsoft.Build.Utilities;
using System.IO;
using System.Diagnostics;

namespace Curse.MSBuild.AssetPackager
{
    public class AssetPackager : ITask
    {
        private class VirtualDirectory
        {
            public string Source { get; set; }
            public string Destination { get; set; }
        }

        private class Asset
        {
            public Asset(XElement element)
            {
                if (element.Attribute("include") != null)
                {
                    string includePath = element.Attribute("include").Value;
                    // Read in the XML file
                    XDocument doc = null;
                    XElement assetPackageLibrary = null;
                    try
                    {
                        doc = XDocument.Load(includePath);

                        assetPackageLibrary = doc.Descendants("assetPackageLibrary").FirstOrDefault();
                        if (assetPackageLibrary == null)
                        {
                            throw new Exception(string.Format("assetPackages element not found in {0}", includePath));
                        }

                    }
                    catch (Exception ex)
                    {
                        throw new Exception("Unable to load asset include file at: " + includePath, ex);
                    }

                    FilePaths = assetPackageLibrary.Elements("asset").Select(x => x.Attribute("filepath").Value).ToArray();
                }
                else
                {
                    FilePaths = new[] { element.Attribute("filepath").Value };
                }
            }

            public string[] FilePaths
            {
                get;
                private set;
            }
        }

        private class PackageData
        {
            public string Name { get; set; }
            public string Type { get; set; }
            public string OutputFilepath { get; set; }
            public string Processing { get; set; }
            public int FileCount { get; set; }
            public Asset[] Assets { get; set; }

            public override string ToString()
            {
                return string.Format("{{ Name = {0}, Type = {1}, OutputFilepath = {2}}", Name, Type);
            }
        }

        public static string FixDirectorySeparators(string path)
        {
            return path.Replace('/', System.IO.Path.DirectorySeparatorChar);
        }

        private static string HandleVirtualPaths(string path, IEnumerable<VirtualDirectory> virtualDirectories)
        {            
            path = FixDirectorySeparators(path);

            if (File.Exists(path) || Directory.Exists(path))
            {                
                return path;
            }

            foreach (VirtualDirectory virtualDirectory in virtualDirectories)
            {
                string virtualPath = FixDirectorySeparators(virtualDirectory.Source) + @"\" + path;                

                if (File.Exists(virtualPath) || Directory.Exists(virtualPath))
                {                    
                    return virtualPath;
                }
            }
            return path;
        }

        public bool Execute()
        {
            
            XDocument doc = XDocument.Load(CobaltConfigFile);
            if (doc == null)
            {
                throw new Exception(string.Format("Can't load XML file {0}", CobaltConfigFile));
            }

            XElement assetPackages = doc.Descendants("assetPackages").FirstOrDefault();
            if (assetPackages == null)
            {
                throw new Exception(string.Format("assetPackages element not found in {0}", CobaltConfigFile));
            }

            XAttribute prefixAttribute = assetPackages.Attribute("prefix");
            string prefix = "";
            if (prefixAttribute != null)
            {
                prefix = prefixAttribute.Value + "/";
            }
            XAttribute destinationFolderAttribute = assetPackages.Attribute("destinationFolder");
            string destinationFolder = "";
            if (destinationFolderAttribute != null)
            {
                destinationFolder = destinationFolderAttribute.Value + "/";
            }

            IEnumerable<PackageData> packages = assetPackages
                .Elements("assetPackage")
                .Select(x =>
                    {
                        PackageData packageData = new PackageData
                        {
                            Name = x.Attribute("name").Value,
                            Type = x.Attribute("type").Value,
                            OutputFilepath = destinationFolder + x.Attribute("outputFilepath").Value,
                            Processing = x.Attribute("processing") != null ? x.Attribute("processing").Value : null,
                            Assets = x.Elements("asset").Select(a => new Asset(a)).ToArray()
                        };

                        int fileCount = 1;
                        if (x.Attribute("fileCount") != null)
                        {
                            int.TryParse(x.Attribute("fileCount").Value, out fileCount);
                        }
                        packageData.FileCount = fileCount;
                        return packageData;
                    });

            XElement virtualDirectoriesElement = doc.Descendants("virtualDirectories").FirstOrDefault();
            IEnumerable<VirtualDirectory> virtualDirectories;
            if (virtualDirectoriesElement != null)
            {
                virtualDirectories = virtualDirectoriesElement
                    .Elements("virtualDirectory")
                    .Select(x => new VirtualDirectory
                    {
                        Source = x.Attribute("source").Value,
                        Destination = x.Attribute("destination").Value
                    });
            }
            else
            {
                virtualDirectories = Enumerable.Empty<VirtualDirectory>();
            }

            foreach (PackageData package in packages)
            {
                Curse.MSBuild.Packer.Packer pack = new Curse.MSBuild.Packer.Packer();
                pack.Verbose = Verbose;
                ICollection<TaskItem> inputFiles = new List<TaskItem>();
                foreach (Asset asset in package.Assets)
                {
                    foreach (string filepath in asset.FilePaths)
                    {
                        inputFiles.Add(new TaskItem(FixDirectorySeparators(RootFolder + "/" + HandleVirtualPaths(prefix + filepath, virtualDirectories))));
                    }
                }
                pack.InputFiles = inputFiles.ToArray();
                pack.OutputFileName = FixDirectorySeparators(RootFolder + "/" + HandleVirtualPaths(prefix + package.OutputFilepath, virtualDirectories));
                pack.OutputFileCount = package.FileCount;
                pack.Processing = package.Processing;
                pack.HostObject = HostObject;
                pack.BuildEngine = BuildEngine;
                pack.AbsolutePathBase = "Content";
                if (!pack.Execute())
                {
                    return false;
                }
            }

            return true;
        }

        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }

        [Required]
        public string CobaltConfigFile { get; set; }
        public bool Verbose { get; set; }
        [Required]
        public string RootFolder { get; set; }
    }
}
