﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text.RegularExpressions;
using Newtonsoft.Json.Linq;

namespace Curse.MSBuild.Chef
{
    public static class SousChef
    {
        static SousChef()
        {

            SousChefConfig.Create("https://sloth01a-dev.curse.io", "adamar", "dev");
            SousChefConfig.Create("https://sloth01a-iad.curse.io", "mcomperda", "iad");
            SousChefConfig.Create("https://sloth01a-dub.curse.io", "mcomperda", "dub");
            SousChefConfig.Create("https://sloth01a-sin.curse.io", "mcomperda", "sin");
           
            System.Net.ServicePointManager.ServerCertificateValidationCallback +=
            delegate {
                return true; // **** Always accept
            };
        }

        public static ChefElasticNode[] GetElasticNodesForEnvironments(IEnumerable<string> environments, string type)
        {
            var nodes = new HashSet<ChefElasticNode>();
            Console.WriteLine("Querying Chef for " + type + " elastic nodes...");
            foreach (var environment in environments)
            {

                var resp = SousChefConfig.MakeRequest<ChefElasticNodeResults>(environment, string.Format("organizations/{1}/data/elasticsearch/{0}-{1}", type, environment));
                foreach (var node in resp.Nodes)
                {
                    node.Value.Name = node.Key;

                    if (!node.Value.IsDataNode && !!node.Value.IsMasterNode)
                    {
                        nodes.Add(node.Value);
                    }
                }
            }
            
            
            return nodes.ToArray();
        }

        public static ChefIISNode[] GetIISNodesForService(IEnumerable<string> environments, string serviceName, bool validateBindings = true)
        {
            var nodes = new HashSet<ChefIISNode>();
            foreach (var environment in environments)
            {
                var nodeNames = GetIISNodesForService(environment, serviceName);
                foreach (var nodeName in nodeNames)
                {
                    nodes.Add(nodeName);
                }
            }

            if (!nodes.Any())
            {
                return new ChefIISNode[0];
            }

            var firstHostName = nodes.First().HostName;

            if (validateBindings && nodes.Any(p => p.HostName != firstHostName))
            {
                Console.WriteLine("Discovered inconsistent host names in config: " +
                                  string.Join(", ", nodes.Select(p => p.HostName)));
                return new ChefIISNode[0];
            }

            return nodes.ToArray();
        }

        public static ChefWindowsNode[] GetWindowsNodesForService(IEnumerable<string> environments, string serviceName, bool validateBindings = true)
        {
            var nodes = new HashSet<ChefWindowsNode>();
            foreach (var environment in environments)
            {
                var nodeNames = GetWindowsNodesForService(environment, serviceName);
                foreach (var nodeName in nodeNames)
                {
                    nodes.Add(nodeName);
                }
            }

            if (!nodes.Any())
            {
                return new ChefWindowsNode[0];
            }

            return nodes.ToArray();
        }


        public static ChefWindowsNode[] GetWindowsNodesForService(string environment, string serviceName)
        {
            environment = environment.ToLowerInvariant();

            Console.WriteLine("Querying Chef for " + environment + " / " + serviceName + "...");
            var resp = SousChefConfig.MakeRequest<ChefWindowsNodeResults>(environment, string.Format("/organizations/{0}/search/node?q=tags:{1}", environment, serviceName));

            if (resp.Total == 0)
            {
                return new ChefWindowsNode[0];
            }

            return resp.Rows.OrderBy(p => p.Name.ToLowerInvariant()).ToArray();
        }

        private static readonly Regex SecureBindingRegex = new Regex(@"^https/\*\:443:(?<HostName>.*?)$", RegexOptions.Compiled);

        public static ChefIISNode[] GetIISNodesForService(string environment, string serviceName)
        {
            environment = environment.ToLowerInvariant();
            var tag = serviceName.Contains("-") ? "client-" + serviceName : string.Format("client-{0}-service", serviceName);
            var bindingName = serviceName.Contains("-") ? serviceName : string.Format("{0}-service", serviceName);

            Console.WriteLine("Querying Chef for " + environment + " / " + tag + "...");
            var resp = SousChefConfig.MakeRequest<ChefSearchResults>(environment, string.Format("/organizations/{0}/search/node?q=tags:{1}", environment, tag));
            
            if (resp.Total == 0)
            {
                return new ChefIISNode[0];
            }

            var nodes = new List<ChefIISNode>();

            foreach (JObject row in resp.Rows)
            {
                var name = (string)row["name"];

                var bindingsVal = FindIISBindings(row, bindingName);
                var preferredBindingVal = FindPreferredIISBinding(row, bindingName);

                var bindings = bindingsVal.Split(',').OrderByDescending(p => p == preferredBindingVal).ThenBy(p => p.Length).ToArray();

                foreach (var binding in bindings)
                {
                    var match = SecureBindingRegex.Match(binding);
                    if (match.Success)
                    {
                        nodes.Add(new ChefIISNode
                        {
                            Environment = environment,
                            Name = name,
                            HostName = match.Groups["HostName"].Value,
                            IsPreferred = binding == preferredBindingVal,
                        });
                        break;
                    }
                }
            }

            return nodes.OrderBy(p => p.Name.ToLowerInvariant()).ToArray();            
        }

        public static string FindIISBindings(JObject row, string serviceName)
        {
            var bindings = GetChefIISKey(row, serviceName, "bindings");

            if (bindings == null)
            {
                Console.WriteLine("Failed to get IIS binding for service :( Press any key to stop the deploy...");
                Console.ReadKey(true);
                throw new Exception("Unable to get IIS bindings!");
            }

            return bindings;            
        }
        public static string FindPreferredIISBinding(JObject row, string serviceName)
        {
            return GetChefIISKey(row, serviceName, "preferred_binding");
        }

        private static List<string> ChefSections = new List<string> { "override", "normal", "default" };

        private static string GetChefIISKey(JObject row, string serviceName, string key)
        {
            foreach (var section in ChefSections)
            {
                try
                {
                    var value = (string)row[section][serviceName]["iis"][key];
                    if (!String.IsNullOrWhiteSpace(value))
                    {
                        return value;
                    }
                }
                catch { }
            }

            return null;
        }

        public static dynamic GetDynamicNode(string environment, string nodeName)
        {
            return SousChefConfig.MakeRequest<object>(environment, string.Format("/organizations/{0}/nodes/{1}", environment, nodeName));
        }
    }
}
