﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Net.Http;
using System.Reflection;
using mattberther.chef;
using Newtonsoft.Json;

namespace Curse.MSBuild.Chef
{
    public class SousChefConfig
    {

        private static readonly Dictionary<string, SousChefConfig> Configurations = new Dictionary<string, SousChefConfig>();
        
        public Uri BaseUrl { get; private set; }
        public string Username { get; private set; }
        public string Environment { get; private set; }
        public string PrivateKey { get; private set; }

        private readonly ChefServer _server;

        private static string GetPrivateKey(string environment)
        {
            var keyPath = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), 
                "ChefApiKey." + environment + ".txt");

            return File.ReadAllText(keyPath);
        }

        private SousChefConfig(string baseUrl, string username, string environment)
        {
            BaseUrl = new Uri(baseUrl);
            Username = username;
            Environment = Environment;
            _server = new ChefServer(BaseUrl);
            PrivateKey = GetPrivateKey(environment);
            Configurations.Add(environment, this);
        }

        public static void Create(string baseUrl, string username, string environment)
        {
            new SousChefConfig(baseUrl, username, environment);
        }

        public static string MakeRequest(string environment, string relativeUrl)
        {
            var config = Configurations[environment];
            return config.MakeRequest(relativeUrl);
        }

        public static T MakeRequest<T>(string environment, string relativeUrl)
        {
            if (!Configurations.ContainsKey(environment))
            {
                throw new InvalidOperationException("Unknown environment: " + environment);
            }
            var config = Configurations[environment];
            var resp = config.MakeRequest(relativeUrl);           
            return JsonConvert.DeserializeObject<T>(resp);
        }

        public static string MakePost(string environment, string relativeUrl, string body)
        {
            var config = Configurations[environment];
            return config.MakePost(relativeUrl, body);
        }

        public string MakeRequest(string relativeUrl)
        {
            var fullUri = new Uri(BaseUrl, relativeUrl);
            Console.WriteLine("Requesting: " + fullUri.ToString());
            var authenticatedRequest = new AuthenticatedRequest(Username, fullUri);
            authenticatedRequest.Sign(PrivateKey);
            var resultContent = _server.SendRequest(authenticatedRequest);
            return resultContent;
        }

        public string MakePost(string relativeUrl, string body)
        {
            var fullUri = new Uri(BaseUrl, relativeUrl);
            var authenticatedRequest = new AuthenticatedRequest(Username, fullUri, new HttpMethod("POST"), body);
            authenticatedRequest.Sign(PrivateKey);
            var resultContent = _server.SendRequest(authenticatedRequest);
            return resultContent;
        }
    }
}
