﻿using System.Diagnostics;
using System.Threading;
using Microsoft.Build.Framework;
using System;
using System.Collections.Concurrent;
using System.IO;
using System.Linq;
using System.Threading.Tasks;
using Curse.MSBuild.Chef;
using Twitch.Arcana;

namespace Curse.MSBuild.Deployment
{
    public class AuthenticatedWindowsServiceDeploy : ITask
    {
        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }

        private string TempFolderPath
        {
            get { return string.Format(@"c:\Windows\Temp\Deploys\{0}{1}", ApplicationName, ApplicationVersion ?? "Unversioned"); }
        }

        private string GetApplicationPath(string node, bool returnLocalPath = false)
        {

            return returnLocalPath ? string.Format(@"c:\Program Files\Curse\{0}{1}", ApplicationName, VersioningDisabled ? "" :  "\\" + ApplicationVersion) :
                string.Format(@"\\{0}\c$\Program Files\Curse\{1}{2}", node, ApplicationName, VersioningDisabled ? "" : "\\" + ApplicationVersion);
        }

        private string GetCurrentApplicationPath(string node, bool returnLocalPath = false)
        {

            return returnLocalPath ? string.Format(@"c:\Program Files\Curse\{0}\Current", ApplicationName) :
                string.Format(@"\\{0}\c$\Progra~1\Curse\{1}\Current", node, ApplicationName);
        }
        
        public string ChefServiceName { get; set; }
        
        [Required]
        public string CompilationMode { get; set; }
        
        public bool Execute()
        {
            var artifactUploadOnly = false;

            if (!AutoCommit && UseS3)
            {
                var resp = DeployUtils.PromptUser("Would you like to build and upload an artifact to S3 without deploying?", new[] { ConsoleKey.Y, ConsoleKey.N });
                artifactUploadOnly = resp == ConsoleKey.Y;
                S3Helper.Initialize();
            }

            if (!artifactUploadOnly && !string.IsNullOrEmpty(ChefServiceName))
            {
                if (string.IsNullOrEmpty(CompilationMode))
                {
                    Console.WriteLine("CompilationMode must be supplied.");
                    return false;
                }

                Console.WriteLine("Looking for nodes included in " + CompilationMode + " deploys...");

                var environments = DeployUtils.GetChefEnvironments(CompilationMode);
                if (environments == null || environments.Length == 0)
                {
                    Console.WriteLine(CompilationMode + " compilations cannot be deployed using the Chef API.");
                    return false;
                }

                Console.WriteLine("Querying Chef for '" + ChefServiceName + "' service nodes in the following environments: " + string.Join(", ", environments));

                var nodes = SousChef.GetWindowsNodesForService(environments, ChefServiceName);
                if (!nodes.Any())
                {
                    Console.WriteLine("Unable to deploy. Chef found no nodes for this service: " + ChefServiceName);
                    return false;
                }

                ServerNames = nodes.Select(p => p.Name).ToArray();                
            }

            if (!artifactUploadOnly && ServerFilteringPrompt)
            {
                ServerNames = DeployUtils.PromptServerNames(ServerNames);
            }

            var s3currentPath = String.Format("{0}/{0}_current.zip", ApplicationName);
            var s3deployPath = String.Format("{0}/{0}_{1:s}.zip", ApplicationName, DateTime.UtcNow);

            DeployStep.Initialize();

            DeployStep.RegisterStep("Calculation Version Number", () => true, () =>
            {
                var path = Path.Combine(ApplicationSourcePath, ApplicationExecutableFileName);
                
                if (!File.Exists(path))
                {
                    Console.WriteLine(@"Unable to determine version number. File is missing: " + path);
                    throw new Exception("Missing file.");
                }

                var versionInfo = FileVersionInfo.GetVersionInfo(path);
                ApplicationVersion = versionInfo.ProductVersion; 

            });

            DeployStep.RegisterStep("Creating Temp Folder", () => true, () =>
            {
                var tempFolder = new DirectoryInfo(TempFolderPath);
                if (!tempFolder.Exists)
                {
                    tempFolder.Create();
                }
            });

            DeployStep.RegisterStep("Replacing Secret Placeholders", () => !SkipSecrets, () =>
            {
                var secretsHelper = SecretsHelper.GetSecretsHelper();
                secretsHelper.ReplaceSecrets(ApplicationSourcePath, CompilationMode);                
            });

            DeployStep.RegisterStep("Zip Application Files", () => true, () =>
            {
                AppZipPath = Path.Combine(TempFolderPath, "Application.zip");
                if (File.Exists(AppZipPath))
                {
                    File.Delete(AppZipPath);
                }

                ZipHelper.CreateZip(ApplicationSourcePath, AppZipPath);
            });

            string appZipUrl = null;
            DeployStep.RegisterStep("Upload artifact to S3", () => UseS3, () =>
            {
                Console.Write("Uploading app zip to S3... ");
                appZipUrl = S3Helper.SaveToS3(s3deployPath, AppZipPath);
                Console.WriteLine("Done");
            });

            DeployStep.RegisterStep("Copy and Unzip App Files", () => !artifactUploadOnly, () =>
            {
                if (UseS3)
                {
                    var stopDeploy = false;
                    var failedNodes = new ConcurrentBag<string>();

                    Parallel.ForEach(ServerNames, node =>
                    {
                        try
                        {
                            Console.WriteLine("Downloading app zip on {0}...", node);
                            using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                            {
                                RemoteScriptHelper.ExecuteRemoteScript(runspace,
                                    string.Format(@"Invoke-WebRequest ""{0}"" -OutFile ""{1}""", appZipUrl, PathHelper.GetRemoteTempFilePath(AppZipPath, node, ApplicationVersion, false)));
                            }

                            Console.WriteLine("Extracting app zip on {0}...", node);
                            string applicationPath = GetApplicationPath(node);
                            DirectoryInfo di = new DirectoryInfo(applicationPath);
                            if (!di.Exists)
                            {
                                di.Create();
                            }

                            using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                            {
                                ZipHelper.UnzipRemotelyWithPowershell(runspace, PathHelper.GetRemoteTempFilePath(AppZipPath, node, ApplicationVersion, true), GetApplicationPath(node, true), node,
                                    AppZipPath);
                            }
                        }
                        catch (Exception ex)
                        {
                            failedNodes.Add(node);
                            Console.WriteLine("Failed on node " + node + ": " + ex.Message);
                            
                        }
                        
                    });


                    if (failedNodes.Any())
                    {

                        Console.WriteLine("Failed on nodes: " + string.Join(", ", failedNodes));
                        var result = PromptUser("Do you want to stop the deploy?", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.N);
                        if (result == ConsoleKey.Y)
                        {
                            throw new Exception("Deploy cancelled!");
                        }

                        ServerNames = ServerNames.Where(p => !failedNodes.Contains(p)).ToArray();
                        Console.WriteLine("Deploy targets updated to: " + string.Join(", ", ServerNames));
                    }
                    
                    Console.WriteLine("App files ready on all nodes.");
                }
                else
                {
                    Parallel.ForEach(ServerNames, node =>
                    {
                        Console.WriteLine("Copying app files to '" + node + "'...");
                        var versionedDirectory = new DirectoryInfo(GetApplicationPath(node));
                        if (!versionedDirectory.Exists)
                        {
                            Console.Write("Creating remote folder at: " + versionedDirectory.FullName + "...");
                            versionedDirectory.Create();
                            Console.WriteLine("Done");
                        }

                        // Determine version of Current
                        var currentPath = GetCurrentApplicationPath(node);
                        var currentDirectory = new DirectoryInfo(currentPath);
                        if (!currentDirectory.Exists)
                        {
                            Console.Write("Creating remote folder at: " + currentDirectory.FullName + "...");
                            currentDirectory.Create();
                            Console.WriteLine("Done");
                        }

                        Console.WriteLine("Copying zip file to production...");
                        ZipHelper.CopyZipToTemp(AppZipPath, node, ApplicationVersion);

                        using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                        {
                            Console.WriteLine("Unzipping zip file to file server...");
                            ZipHelper.UnzipRemotelyWithPowershell(runspace, PathHelper.GetRemoteTempFilePath(AppZipPath, node, ApplicationVersion, true), GetApplicationPath(node, true), node);
                        }
                    });
                }
            });

            DeployStep.RegisterStep("Commit Deploy", () => !artifactUploadOnly, CommitDeploy);

            if (UseS3)
            {
                DeployStep.RegisterStep("Flag As Current Build", () => true, () =>
                {
                    if (PromptUser("Flag as current build?", new[] { ConsoleKey.Y, ConsoleKey.N }, ConsoleKey.Y) == ConsoleKey.Y)
                    {
                        S3Helper.CopyWithinS3(s3deployPath, s3currentPath);
                    }
                });
            }

            var success = DeployStep.RunAll();
            return success;
        }

        public bool ConfirmCommitToSecondaries { get; set; }

        private void CommitDeploy()
        {
            var counter = 0;

            foreach (var node in ServerNames)
            {
                if (ConfirmCommitToSecondaries)
                {
                    PromptUser("Press enter to commit to " + node, new[] {ConsoleKey.Enter}, ConsoleKey.Enter);
                }

                Console.WriteLine("Commiting to " + node + " (" + (++counter) + " of " + ServerNames.Length + ")");

                SensuHelper.SilenceWorkerAlerts(node, DeployUtils.GetChefEnvironments(CompilationMode));

                using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                {

                    var alreadyInstalled = RemoteScriptHelper.IsServiceInstalled(runspace, ApplicationName);
                    if (alreadyInstalled)
                    {
                        Console.WriteLine("Stopping " + ApplicationName);

                        if (!RemoteScriptHelper.StopService(runspace, ApplicationName))
                        {
                            Console.WriteLine("Failed to stop service!");

                            if (!AutoCommit)
                            {
                                Console.ReadKey();
                            }

                            continue;
                        }
                    }

                    var nodeDelaySeconds = NodeDelay > 0 ? NodeDelay : 20;
                    var nodeDelay = TimeSpan.FromSeconds(nodeDelaySeconds);
                    Console.WriteLine("Waiting " + nodeDelay.TotalSeconds + " seconds for connections to close...");
                    Thread.Sleep(nodeDelay);

                    if (!alreadyInstalled)
                    {
                        Console.WriteLine("Installing " + ApplicationName);
                        WindowsServiceHelper.InstallService(node, ApplicationName, Path.Combine(GetCurrentApplicationPath(node, true), ApplicationExecutableFileName));
                    }

                    var newFolderPath = GetApplicationPath(node, true);
                    var currentFolderPath = GetCurrentApplicationPath(node, true);

                    if (!UpdateCurrentFolder)
                    {
                        // Rename the current 'Current' folder to a timestamp
                        var script = @"Rename-Item -path """ + currentFolderPath + @""" " + DateTime.UtcNow.Ticks;
                        Console.WriteLine(script);
                        RemoteScriptHelper.ExecuteRemoteScript(runspace, script);

                        // Rename the versioned folder to 'Current'
                        script = @"Rename-Item -path """ + newFolderPath + @""" Current";
                        Console.WriteLine(script);
                        RemoteScriptHelper.ExecuteRemoteScript(runspace, script);
                    }
                    else
                    {
                        Console.WriteLine("Updating current folder rather than replacing it.");
                        RemoteScriptHelper.ExecuteRemoteScript(runspace, String.Format(@"Copy-Item -Recurse -Force ""{0}\*"" -Destination ""{1}""", newFolderPath, currentFolderPath));

                        Console.WriteLine("Deleting deploy folder.");
                        RemoteScriptHelper.ExecuteRemoteScript(runspace, String.Format(@"Remove-Item -Recurse -Force ""{0}""", newFolderPath));
                    }

                    Console.WriteLine("Starting " + ApplicationName);
                    RemoteScriptHelper.StartService(runspace, ApplicationName);
                    if ((counter == 1 || ConfirmCommitToSecondaries) && !AutoCommit)
                    {
                        Console.WriteLine("Finished deployment to '" + node + "' Press ENTER to continue rollout.");
                        Console.ReadLine();
                    }

                    Console.WriteLine("Committed to " + node);
                }
            }
        }

        protected ConsoleKey PromptUser(string promptText, ConsoleKey[] options, ConsoleKey defaultChoice)
        {
            if (AutoCommit)
            {
                Console.WriteLine("Using default choice: " + defaultChoice);
                return defaultChoice;
            }

            return DeployUtils.PromptUser(promptText, options);
        }

        public string AppZipPath;

        [Required]
        public string ApplicationName { get; set; }

        [Required]
        public string ApplicationVersion { get; set; }

        [Required]
        public bool VersioningDisabled { get; set; }

        [Required]
        public string ApplicationSourcePath { get; set; }

        [Required]
        public bool ServerFilteringPrompt { get; set; }

        [Required]
        public string[] ServerNames { get; set; }
        
        [Required]
        public string ApplicationExecutableFileName { get; set; }

        public bool AutoCommit { get; set; }

        public bool UsePowershellForRemoteCommands { get; set; }

        public bool SkipSecrets { get; set; }

        public bool UseS3 { get; set; }

        public bool UpdateCurrentFolder { get; set; }

        public int NodeDelay { get; set; }
    }
}
