﻿using Microsoft.Build.Framework;
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;

namespace Curse.MSBuild.Deployment
{
    public class ConsoleAppDeploy : ITask
    {
        public IBuildEngine BuildEngine { get; set; }
        public ITaskHost HostObject { get; set; }

        private string TempFolderPath
        {
            get { return string.Format(@"c:\Windows\Temp\Deploys\{0}{1}", ApplicationName, ApplicationVersion ?? "Unversioned"); }
        }

        private string GetApplicationPath(string node, bool returnLocalPath = false)
        {

            return returnLocalPath
                ? string.Format(@"{2}:\Applications\{0}{1}", ApplicationName,
                    VersioningDisabled ? "" : "\\" + ApplicationVersion, DriveLetter)
                : string.Format(@"\\{0}\{3}$\Applications\{1}{2}", node, ApplicationName,
                    VersioningDisabled ? "" : "\\" + ApplicationVersion, DriveLetter);
        }

        public bool Execute()
        {
            DeployStep.Initialize();

            DeployStep.RegisterStep("Validating Configuration", () => true, () =>
            {
                FileInfo fi = new FileInfo(@"C:\Program Files (x86)\PsExec\psexec.exe");

                if (!fi.Exists)
                {
                    Console.WriteLine(@"You are missing a local copy of PsExec. Please put it at C:\Program Files (x86)\PsExec\psexec.exe");
                    throw new Exception("Missing file.");
                }

            });

            DeployStep.RegisterStep("Creating Temp Folder", () => true, () =>
            {
                DirectoryInfo tempFolder = new DirectoryInfo(TempFolderPath);
                if (!tempFolder.Exists)
                {
                    tempFolder.Create();
                }
            });

            
            DeployStep.RegisterStep("Zip Application Files", () => true, () =>
            {
                AppZipPath = Path.Combine(TempFolderPath, "Application.zip");
                if (File.Exists(AppZipPath))
                {
                    File.Delete(AppZipPath);
                }

                ZipHelper.CreateZip(ApplicationSourcePath, AppZipPath);
            });

            DeployStep.RegisterStep("Copy and Unzip App Files", () => true, () =>
            {
                foreach (var node in ServerNames)
                {
                    DirectoryInfo di = new DirectoryInfo(GetApplicationPath(node));
                    if (!di.Exists)
                    {
                        Console.Write("Creating remote folder at: " + di.FullName + "...");
                        di.Create();
                        Console.WriteLine("Done");
                    }

                    Console.WriteLine("Copying static zip file to production...");
                    ZipHelper.CopyZipToTemp(AppZipPath, node, ApplicationVersion);

                    Console.WriteLine("Unzipping zip file to file server...");
                    if (UsePowershellForRemoteCommands)
                    {
                        using (var runspace = RemoteScriptHelper.CreateRunspace(node))
                        {
                            ZipHelper.UnzipRemotelyWithPowershell(runspace, PathHelper.GetRemoteTempFilePath(AppZipPath, node, ApplicationVersion, true), GetApplicationPath(node, true), node);
                        }
                    }
                    else
                    {
                        ZipHelper.UnzipRemotely(PathHelper.GetRemoteTempFilePath(AppZipPath, node, ApplicationVersion, true), GetApplicationPath(node, true), node);
                    }
                }
            });


            return DeployStep.RunAll();
        }

        public string AppZipPath;

        [Required]
        public string ApplicationName { get; set; }

        [Required]
        public string ApplicationVersion { get; set; }

        [Required]
        public bool VersioningDisabled { get; set; }

        [Required]
        public string ApplicationSourcePath { get; set; }

        [Required]
        public string[] ServerNames { get; set; }

        [Required]
        public string DriveLetter { get; set; }

        public bool UsePowershellForRemoteCommands { get; set; }
    }
}
