﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections.ObjectModel;
using System.IO;
using System.Text.RegularExpressions;
using System.ComponentModel;

namespace Curse.MSBuild.Deployment
{
    public class HostEntry : IHostEntry
    {
        private string mRawString;
        private string mHostname;
        private bool mIsEnabled;

        public string RawString { get; set; }
        public bool IsComment { get; set; }
        public bool IsEnabled { get; set; }        
        public string Comment { get; set; } // If line contains # but is not at the start, anything after is a comment
        public string IpAddress { get; set; }        
        public string HostName { get; set; }
        public string Group { get; set; }
        public int LineNumber { get; set; }

        public HostEntry()
        {
            
        }

        public HostEntry(string hostName, string ipAddress, int lineNumber)
        {
            IpAddress = ipAddress;
            HostName = hostName;
            IsComment = false;
            IsEnabled = true;
            LineNumber = lineNumber;
        }

        public object Clone() // ICloneable implementation
        {
            HostEntry mc = this.MemberwiseClone() as HostEntry;
            return mc;
        }
       
    }
    public class HeaderEntry : IHostEntry
    {
        public string Group { get; set; }
        private string mRawString;
        public string RawString
        {
            get
            {
                return mRawString;
            }
            set
            {
                if (mRawString != value)
                {
                    mRawString = value;
                }
            }
        }
        public int LineNumber { get; set; }
        public object Clone() // ICloneable implementation
        {
            HeaderEntry mc = this.MemberwiseClone() as HeaderEntry;
            return mc;
        }
    }
    public interface IHostEntry : ICloneable
    {
        string RawString { get; set; }
        int LineNumber { get; set; }
    }
    public class HostFile
    {
        public List<IHostEntry> Entries
        {
            get;
            set;
        }
        public List<IHostEntry> Comments
        {
            get;
            set;
        }
        public string HostPath
        {
            get;
            set;
        }
        public void Save()
        {
            List<IHostEntry> MergedEntries = new List<IHostEntry>();
            foreach (IHostEntry commentEntry in Comments)
            {
                MergedEntries.Add(commentEntry);
            }
            foreach (IHostEntry hostEntry in Entries)
            {
                MergedEntries.Add(hostEntry);
            }

            foreach (IHostEntry entry in MergedEntries)
            {
                if (entry is HostEntry)
                {
                    if (((HostEntry)entry).IsComment) { continue; }

                    string rawString = ((HostEntry)entry).IpAddress + "\t" + ((HostEntry)entry).HostName;
                    if ((((HostEntry)entry).Comment != null) && (((HostEntry)entry).Comment != ""))
                    {
                        rawString += "\t#" + ((HostEntry)entry).Comment;
                    }

                    if (!((HostEntry)entry).IsEnabled)
                    {
                        rawString = "#" + rawString;
                    }
                    entry.RawString = rawString;
                }
                else if (entry is HeaderEntry)
                {
                    if (((HeaderEntry)entry).Group != null)
                    {
                        entry.RawString = "### " + ((HeaderEntry)entry).Group;
                    }
                }
            }


            List<IHostEntry> SortedEntries = MergedEntries.OrderBy(p => p.LineNumber).ToList();
            using (StreamWriter outfile = new StreamWriter(HostPath))
            {
                foreach (IHostEntry outputEntry in SortedEntries)
                {                  
                    outfile.WriteLine(outputEntry.RawString);
                }
            }
        }

        protected HostFile(string hostPath)
        {

            HostPath = hostPath;

            // Backup your first host file just in case.
            if (!File.Exists(HostPath + ".bak"))
            {
                File.Copy(HostPath, HostPath + ".bak");
            }

            Entries = new List<IHostEntry>();
            Comments = new List<IHostEntry>();


            int lineNum = 0;

            string lastGroup = null;

            using (StreamReader sr = new StreamReader(hostPath))
            {
                string line;

                while ((line = sr.ReadLine()) != null)
                {
                    if ((line == "#") || (line.StartsWith("# ")) || line.StartsWith("#\t"))
                    {
                        HostEntry ce = new HostEntry();
                        ce.RawString = line;
                        ce.LineNumber = lineNum;
                        ce.IsComment = true;
                        Comments.Add(ce);
                        lineNum++;
                    }
                    else if (line == "")
                    {
                        lastGroup = null;
                        if (Entries.Count != 0)
                        {
                            HeaderEntry headerEntry = new HeaderEntry();
                            headerEntry.RawString = line;
                            headerEntry.LineNumber = lineNum;
                            Entries.Add(headerEntry);
                        }
                        lineNum++;
                    }
                    else if (line.StartsWith("### "))
                    {
                        lastGroup = line.Remove(0, 4);

                        HeaderEntry headerEntry = new HeaderEntry();
                        headerEntry.RawString = line;
                        headerEntry.LineNumber = lineNum;
                        headerEntry.Group = lastGroup;
                        Entries.Add(headerEntry);

                        lineNum++;
                    }
                    else if ((line.StartsWith("# ") == false) && (line != ""))
                    {
                        Regex regx = new Regex("[^\t]+|\t(?=\t)|\t$");
                        MatchCollection matches = regx.Matches(line);

                        if (matches.Count < 2)
                        {
                            Regex regex = new Regex("[^ ]+| (?= )| $");
                            matches = regex.Matches(line);
                            if (matches.Count < 2) { continue; }
                        }

                        HostEntry he = new HostEntry();
                        he.IsComment = false;

                        if (matches[0].Value.StartsWith("#"))
                        {
                            he.IpAddress = matches[0].Value.Remove(0, 1);
                            he.IsEnabled = false;
                        }
                        else
                        {
                            he.IpAddress = matches[0].Value;
                            he.IsEnabled = true;
                        }

                        if (lastGroup != null)
                        {
                            he.Group = lastGroup;
                        }

                        string pointsTo = "";

                        foreach (Match match in matches)
                        {
                            if (matches[0] == match) { continue; }
                            if (match.Value != "\t" && match.Value != "" && match.Value != " ")
                            {
                                string matchValue = match.Value;
                                if (matchValue.StartsWith("# "))
                                {
                                    matchValue = matchValue.Replace("# ", "");
                                    he.Comment = matchValue;
                                }
                                else if (matchValue.StartsWith("#"))
                                {
                                    matchValue = matchValue.Replace("#", "");
                                    he.Comment = matchValue;
                                }
                                else if (pointsTo == "")
                                {
                                    pointsTo = match.Value;
                                }
                            }
                        }
                        if (pointsTo == "" && he.Comment != null && he.IsEnabled)
                        {
                            pointsTo = he.Comment;
                        }

                        he.HostName = pointsTo;
                        he.RawString = line;
                        he.LineNumber = lineNum;

                        Entries.Add(he);
                        lineNum++;
                    }

                }
            }
            
        }        

        public static HostFile Parse(string hostPath = null)
        {            
            if (hostPath == null)
            {                
                hostPath = Environment.GetFolderPath(Environment.SpecialFolder.Windows) + @"\System32\drivers\etc\hosts";
            }

            return new HostFile(hostPath);
        }

        public void AddEntry(string hostName, string ipAddress)
        {
            HostEntry hostEntry = new HostEntry(hostName, ipAddress, Entries.Count + Comments.Count + 1);
            Entries.Add(hostEntry);
        }

        public void AddOrReplaceEntry(string hostName, string ipAddress, bool makeActive = true)
        {
            HostEntry existingEntry = (HostEntry)Entries.Where(p => p is HostEntry).FirstOrDefault(p => string.Equals((p as HostEntry).HostName, hostName));

            if (existingEntry == null)
            {
                AddEntry(hostName, ipAddress);
            }
            else
            {
                existingEntry.IpAddress = ipAddress;
                if (makeActive)
                {
                    existingEntry.IsEnabled = true;
                }
            }
        }

        public void RemoveEntry(string hostName)
        {
            HostEntry existingEntry = (HostEntry)Entries.Where(p => p is HostEntry).FirstOrDefault(p => string.Equals((p as HostEntry).HostName, hostName));
            if (existingEntry != null)
            {
                Entries.Remove(existingEntry);
            }
        }
    }
}
